% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evo.dir.R
\name{evo.dir}
\alias{evo.dir}
\title{Phylogenetic vector analysis of phenotypic change}
\usage{
evo.dir(RR,angle.dimension=c("rates","phenotypes"),
y.type=c("original","RR"),y=NULL,pair.type=c("node","tips"),pair=NULL,
node=NULL,random=c("yes","no"),nrep=100)
}
\arguments{
\item{RR}{an object produced by \code{\link{RRphylo}}.}

\item{angle.dimension}{specifies whether vectors of \code{"rates"} or \code{"phenotypes"} are used.}

\item{y.type}{must be indicated when \code{angle.dimension = "phenotypes"}. If \code{"original"}, it uses the phenotypes as provided by the user, if \code{"RR"} it uses \code{RR$predicted.phenotypes}.}

\item{y}{specifies the phenotypes to be provided if \code{y.type = "original"}.}

\item{pair.type}{either \code{"node"} or \code{"tips"}. Angles are computed between each possible couple of species descending from a specified node (\code{"node"}), or between a given couple of species (\code{"tips"}).}

\item{pair}{species pair to be specified if \code{pair.type = "tips"}. It needs to be written as in the example below.}

\item{node}{node number to be specified if \code{pair.type = "node"}. Notice the node number must refer to the dichotomic version of the original tree, as produced by \code{RRphylo}.}

\item{random}{whether to perform randomization test (\code{"yes"}/\code{"no"}).}

\item{nrep}{number of replications must be indicated if \code{random = "yes"}. It is set at 100 by default.}
}
\value{
Under all specs, \code{evo.dir} returns a 'list' object. The length of the list is one if \code{pair.type = "tips"}.  If \code{pair.type = "node"}, the list is as long as the number of all possible species pairs descending from the node. Each element of the list contains:

\strong{angle.path.A} angle of the resultant vector of species A to MRCA

\strong{vector.size.species.A} size of the resultant vector of species A to MRCA

\strong{angle.path.B} angle of the resultant vector of species B to MRCA

\strong{vector.size.species.B} size of the resultant vector of species B to MRCA

\strong{angle.between.species.to.mrca} angle between the species paths resultant vectors to the MRCA

\strong{angle.between.species} angle between species vectors (as they are, without computing the path)

\strong{MRCA} the node identifying the most recent common ancestor of A and B

\strong{angle.direction.A} angle of the vector of species A (as it is, without computing the path) to a fixed reference vector (the same for all species)

\strong{vec.size.direction.A} size of the vector of species A

\strong{angle.direction.B} angle of the vector of species B (as it is, without computing the path) to a fixed reference vector (the same for all species)

\strong{vec.size.direction.B} size of the vector of species B

If \code{random = "yes"}, results also include p-values for the angles.
}
\description{
This function quantifies direction, size and rate of evolutionary change of multivariate traits along node-to-tip paths and between species.
}
\details{
The way \code{evo.dir} computes vectors depends on whether phenotypes or rates are used as variables. \code{\link{RRphylo}} rates along a path are aligned along a chain of ancestor/descendant relationships. As such, each rate vector origin coincides to the tip of its ancestor, and the resultant of the path is given by vector addition. In contrast, phenotypic vectors are computed with reference to a common origin (i.e. the consensus shape in a geometric morphometrics). In this latter case, vector subtraction (rather than addition) will define the resultant of the evolutionary direction.  It is important to realize that resultants could be at any angle even if the species (the terminal vectors) have similar phenotypes, because path resultants, rather than individual phenotypes, are being contrasted. However, the function also provides the angle between individual phenotypes as 'angle.between.species'. To perform randomization test (\code{random = "yes"}), the evolutionary directions of the two species are collapsed together. Then, for each variable, the median is found, and random paths of the same size as the original paths are produced sampling at random from the 47.5th to the 52.5th percentile around the medians. This way, a random distribution of angles is obtained under the hypothesis that the two directions are actually parallel. The 'angle.direction' represents the angle formed by the species phenotype and a vector of 1s (as long as the number of variables representing the phenotype). This way, each species phenotype is contrasted to the same vector. The 'angle.direction' values could be inspected to test whether individual species phenotypes evolve towards similar directions.
}
\examples{
  data("DataApes")
  DataApes$PCstage->PCstage
  DataApes$Tstage->Tstage

  RRphylo(tree=Tstage,y=PCstage)->RR
# Case 1. Without performing randomization test

 # Case 1.1 Computing angles between rate vectors
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="rates",pair.type="node",node=72 ,
    random="no")
  # for a given couple of species
    evo.dir(RR,angle.dimension="rates",pair.type="tips",
    pair= c("Sap_1","Tro_2"),random="no")

 # Case 1.2 computing angles between phenotypic vectors provided by the user
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="phenotypes",y.type="original",
    y=PCstage,pair.type="node",node=72,random="no")
  # for a given couple of species
    evo.dir(RR,angle.dimension="phenotypes",y.type="original",
    y=PCstage,pair.type="tips",pair=c("Sap_1","Tro_2"),random="no")

 # Case 1.3 computing angles between phenotypic vectors produced by "RRphylo"
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="phenotypes",y.type="RR",
    pair.type="node",node=72,random="no")
  # for a given couple of species
    evo.dir(RR,angle.dimension="phenotypes",y.type="RR",
    pair.type="tips",pair=c("Sap_1","Tro_2"),random="no")

\donttest{
# Case 2. Performing randomization test

 # Case 2.1 Computing angles between rate vectors
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="rates",pair.type="node",node=72 ,
    random="yes",nrep=100)

  # for a given couple of species
    evo.dir(RR,angle.dimension="rates",pair.type="tips",
    pair= c("Sap_1","Tro_2"),random="yes",nrep=100)

 # Case 2.2 computing angles between phenotypic vectors provided by the user
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="phenotypes",y.type="original",
    y=PCstage,pair.type="node",node=72,random="yes",nrep=100)

  # for a given couple of species
    evo.dir(RR,angle.dimension="phenotypes",y.type="original",
    y=PCstage,pair.type="tips",pair=c("Sap_1","Tro_2"),random="yes",nrep=100)

 # Case 2.3 computing angles between phenotypic vectors produced by "RRphylo"
  # for each possible couple of species descending from node 72
    evo.dir(RR,angle.dimension="phenotypes",y.type="RR",
    pair.type="node",node=72,random="yes",nrep=100)

  # for a given couple of species
    evo.dir(RR,angle.dimension="phenotypes",y.type="RR",
    pair.type="tips",pair=c("Sap_1","Tro_2"),random="yes",nrep=100)
    }
}
\author{
Pasquale Raia, Silvia Castiglione, Carmela Serio, Alessandro Mondanaro, Marina Melchionna, Mirko Di Febbraro, Antonio Profico, Francesco Carotenuto
}
