#' @title
#' Check for possible API user privilege issues

#' @description
#' Check for potential user access privilege issues and provide an appropriate
#' warning message. This can occur when metadata forms/field names do not appear
#' in a database export.
#'
#' @return
#' A helpful error message alerting the user to check their API privileges.
#'
#' @importFrom rlang .data
#' @importFrom dplyr filter select group_by summarise
#' @importFrom tidyr pivot_wider
#' @importFrom cli cli_warn
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#' @param db_metadata The REDCap metadata output generated by \code{REDCapR::redcap_metadata_read()$data}
#'
#' @keywords internal

check_user_rights <- function(db_data,
                              db_metadata) {
  missing_db_metadata <- db_metadata %>% # nolint: object_usage_linter
    filter(!.data$field_name_updated %in% names(db_data)) %>%
    select("field_name_updated", "form_name") %>%
    group_by(.data$form_name) %>%
    summarise(fields = list(.data$field_name_updated))

  cli_warn(
    c("Instrument name{?s} {missing_db_metadata$form_name} detected in metadata,
        but not found in the database export.",
      "i" = "This can happen when the user privileges are not set to allow
        exporting certain instruments via the API. The following variable{?s}
        are affected: {unlist(missing_db_metadata$fields)}"
    ),
    class = c("redcap_user_rights", "REDCapTidieR_cond")
  )
}

#' @title
#' Check for instruments that have both repeating and non-repeating structure
#'
#' @description
#' Check for potential instruments that are given both repeating and
#' nonrepeating structure. \code{REDCapTidieR} does not support database
#' structures built this way.
#'
#' @return
#' A helpful error message alerting the user to existence of an instrument
#' that was designated both as repeating and non-repeating.
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#'
#' @importFrom dplyr %>% select mutate case_when
#' @importFrom purrr map2
#' @importFrom tidyselect any_of
#' @importFrom cli cli_abort
#'
#' @keywords internal


check_repeat_and_nonrepeat <- function(db_data) {
  # This check function looks for potential repeat/nonrepeat behavior using the
  # steps below:
  # 1) Define standard columns that don't need checking and remove those from
  #    analysis (i.e. "safe columns").
  # 2) Create a dummy column for each remaining column and use case_when to
  #    to assign whether the column demonstrates repeating, nonrepeating, or
  #    indeterminate behavior per row. Indeterminate would be the case for
  #    data not yet filled out.
  # 3) Use a mapping function to determine if any dummy columns contain both
  #    "repeating" AND "nonrepeating" declarations, if so error out.

  # Step (1)
  safe_cols <- c(
    names(db_data)[1], "redcap_event_name",
    "redcap_repeat_instrument", "redcap_repeat_instance"
  )

  # Step (2)
  check_data <- db_data %>%
    mutate(
      across(
        .cols = -any_of(safe_cols),
        .names = "{.col}_repeatingcheck",
        .fns = ~ case_when(
          !is.na(.x) & !is.na(redcap_repeat_instrument) ~ "repeating",
          !is.na(.x) & is.na(redcap_repeat_instrument) ~ "nonrepeating",
          TRUE ~ NA_character_
        )
      )
    )


  # Step (3)
  repeat_nonrepeat_error <- function(check_data, names) { # nolint: object_name_linter

    rep <- gsub(pattern = "_repeatingcheck", replacement = "", x = names) # nolint: object_name_linter

    if ("repeating" %in% check_data &&
      "nonrepeating" %in% check_data) {
      cli_abort(c("x" = "Instrument detected that has both repeating and
      nonrepeating instances defined in the project: {rep}"),
        class = c("repeat_nonrepeat_instrument", "REDCapTidieR_cond")
      )
    }
  }

  purrr::map2(
    .x = check_data %>% select(ends_with("_repeatingcheck")),
    .y = check_data %>% select(ends_with("_repeatingcheck")) %>% names(),
    .f = ~ repeat_nonrepeat_error(.x, .y)
  )
}

#' @title
#' Check that a supplied REDCap database is populated

#' @description
#' Check for potential outputs where metadata is present, but \code{nrow} and
#' \code{ncol} equal `0`. This causes \code{multi_choice_to_labels} to fail, but
#' a helpful error message should be provided.
#'
#' @return
#' A helpful error message alerting the user to check their API privileges.
#'
#' @param db_data The REDCap database output generated by
#' \code{REDCapR::redcap_read_oneshot()$data}
#'
#' @importFrom cli cli_abort
#'
#' @keywords internal

check_redcap_populated <- function(db_data) {
  if (ncol(db_data) == 0) {
    cli_abort(
      "The REDCap API did not return any data. This can happen when there are no
      data entered or when the access isn't configured to allow data export
      through the API.",
      class = c("redcap_unpopulated", "REDCapTidieR_cond")
    )
  }
}


#' @title
#' Check that all requested instruments are in REDCap project metadata
#'
#' @description
#' Provide an error message when any instrument names are passed to
#' \code{read_redcap()} that do not exist in the project metadata.
#'
#' @return
#' An error message listing the requested instruments that don't exist
#'
#' @importFrom cli cli_abort
#'
#' @param db_metadata The metadata file read by
#' \code{REDCapR::redcap_metadata_read()}
#' @param forms The character vector of instrument names passed to
#' \code{read_redcap()}
#'
#' @keywords internal
check_forms_exist <- function(db_metadata, forms) {
  missing_forms <- setdiff(forms, unique(db_metadata$form_name))

  if (length(missing_forms) > 0) {
    cli_abort(
      c("x" = "Instrument{?s} {missing_forms} {?does/do} not exist in REDCap
        project"),
      class = c("form_does_not_exist", "REDCapTidieR_cond")
    )
  }
}

#' @title
#' Check that a supertibble contains \code{redcap_data} and
#' \code{redcap_metadata} fields
#'
#' @description
#' Provide an error message when a tibble is missing \code{redcap_data} or
#' \code{redcap_metadata}
#'
#' @importFrom cli cli_abort
#'
#' @param supertbl a supertibble
#'
#' @return
#' An error message indicating that the required columns are missing
#'
#' @keywords internal
check_req_labelled_fields <- function(supertbl) {
  # Check for presence of req fields
  req_fields <- c("redcap_data", "redcap_metadata")
  missing_fields <- setdiff(req_fields, colnames(supertbl))

  # If any are missing give an error message
  if (length(missing_fields) > 0) {
    cli_abort(
      c(
        "!" = "{.arg supertbl} must contain {.code {req_fields}}",
        "x" = "You are missing {.code {missing_fields}}"
      ),
      class = c("missing_req_labelled_fields", "REDCapTidieR_cond"),
      # pass along the fields that were missing as metadata
      missing_fields = missing_fields
    )
  }
}

#' @title
#' Check that all metadata tibbles within a supertibble contain
#' \code{field_name} and \code{field_label} columns
#'
#' @importFrom purrr map map_int
#' @importFrom dplyr %>% filter
#' @importFrom cli cli_abort
#'
#' @param supertbl a supertibble containing a \code{redcap_metadata} column
#'
#' @return
#' an error message alerting that instrument metadata is incomplete
#'
#' @keywords internal
check_req_labelled_metadata_fields <- function(supertbl) {
  req_fields <- c("field_name", "field_label") # nolint: object_usage_linter

  # map over each metadata tibble and return list element with missing fields
  missing_fields <- supertbl$redcap_metadata %>%
    map(~ setdiff(req_fields, colnames(.)))

  # If any missing fields were found error
  if (length(unlist(missing_fields)) > 0) {
    # Build error message bullets of the form:
    # x: {form} is missing {missing fields}
    msg_data <- tibble(missing_fields = missing_fields)

    # Instrument names to use in message. Use redcap_form_name if available
    # but don't assume it's in the data
    if ("redcap_form_name" %in% colnames(supertbl)) {
      msg_data$form <- supertbl$redcap_form_name
    } else {
      msg_data$form <- paste0(
        "supertbl$redcap_metadata[[", seq_along(missing_fields), "]]"
      )
    }

    # Drop rows without missing fields
    msg_data <- msg_data %>%
      filter(map_int(.data$missing_fields, length) > 0)

    # Create vector of messages and apply 'x' label
    msg <- paste0(
      "{.code {msg_data$form[[", seq_len(nrow(msg_data)), "]]}} ",
      "is missing {.code {msg_data$missing_fields[[", seq_len(nrow(msg_data)), "]]}}"
    )

    names(msg) <- rep("x", length(msg))

    # Prepend note about required fields
    msg <- c(
      "!" = "All elements of {.arg supertbl$redcap_metadata} must contain {.code {req_fields}}",
      msg
    )

    cli_abort(
      msg,
      class = c("missing_req_labelled_metadata_fields", "REDCapTidieR_cond")
    )
  }
}
