#' @title Mode Selection by Stepwise Regression
#' @description Mode selection by Stepwise Regression. The purpose of this function
#' is to select the part modes with similar characteristics to the observed time series
#' from the modes generated by the quantum walk. And it is based on the linear model.
#' @usage qwdap.sws(real, ctqw, index, select_method, plotting)
#' @param real the real series observed.
#' @param ctqw the 'CTQW' object.
#' @param index the index of the data for mode selection.
#' @param select_method choose a stepwise method.
#' @param plotting whether to plot.
#'
#' @return a 'QWMS' object.
#' 
#' @details The 'QWMS' object include the original time series and the modes generated by
#' quantum walks.
#' 
#' @import StepReg
#' @export
#'
#' @examples
#' data("traffic.qw")
#' data("trafficflow")
#' res.sws <- qwdap.sws(trafficflow,traffic.qw,1,"bidirection",TRUE)
qwdap.sws<-function(real,ctqw,index,select_method = c("forward","backward","bidirection"),
                    plotting = FALSE){
  # library(StepReg)
  # get the data
  # data_y = as.data.frame(data_y)
  # data_x = as.data.frame(data_x)
  # # dot need reshape
  # if(p_reshape){
  #   data_x = as.data.frame(matrix(data_x[[1]],nrow = length(data_y[[1]]),byrow = FALSE))
  # }
  if(class(ctqw)!="CTQW"){
    print("The parameter 'ctqw' is not a 'CTQW' object.")
    return()
  }
  if(index<1||index>length(real)){
    print("The parameter 'index' is out of range.")
    return()
  }
  if(nrow(real)!=ctqw$lens){
    print("The row of 'real' is not equal to the 'lens' of 'ctqw'.")
    return()
  }
  if(!is.data.frame(real)){
    real = as.data.frame(real)
  }
  proc.data <- cbind(real[index],as.data.frame(ctqw$ctqw[,index,]))
  my_lm = paste(colnames(proc.data)[1],"~",paste(colnames(proc.data)[-1],collapse = " + "))
  res <- stepwise(formula=as.formula(my_lm),data=proc.data,selection = select_method, select = "AIC")
  if(plotting){
    selected_vars = as.matrix(res$Process[-1,"AIC"])
    rownames(selected_vars)=res$Process[-1,"EnteredEffect"]
    plot(selected_vars,xaxt="n",type="l",lwd=2,col=1,xlab="Parameters",ylab="Index value")
    axis(side=1,at=res$Process[-1,"Step"],labels = as.vector(res$Process[-1,"EnteredEffect"]))
  }
  # if(p_reshape){
  #   return(list(data_x,res))
  # }else{
  #   return(res)
  # }
  res<-list(real=as.matrix(real[index]), ctqw=ctqw$ctqw[,index,], index = index,
            method = select_method, variate = res$Varaibles[-1], importance = res$Process[-1,"AIC"])
  res<-structure(res,class="QWMS")
  return(res)
}
