% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PortfolioOptimProjection.R
\name{PortfolioOptimProjection}
\alias{PortfolioOptimProjection}
\title{Portfolio optimization which finds an optimal portfolio with the smallest distance to a benchmark.}
\usage{
PortfolioOptimProjection (dat, portfolio_return,
risk=c("CVAR","DCVAR","LSAD","MAD"), alpha=0.95, bvec,
Aconstr=NULL, bconstr=NULL, LB=NULL, UB=NULL, maxiter=500, tol=1e-7)
}
\arguments{
\item{dat}{Time series of returns data; dat = cbind(rr, pk), where \eqn{rr} is an array (time series) of asset returns,
for \eqn{n} returns and \eqn{k} assets it is an array with \eqn{\dim(rr) = (n, k)},    
\eqn{pk} is a vector of length \eqn{n} containing probabilities of returns.}

\item{portfolio_return}{Target portfolio return.}

\item{risk}{Risk measure chosen for optimization; one of "CVAR", "DCVAR", "LSAD", "MAD", where
"CVAR" -- denotes Conditional Value-at-Risk (CVaR),
"DCVAR" -- denotes deviation CVaR,
"LSAD" -- denotes Lower Semi Absolute Deviation,
"MAD" -- denotes Mean Absolute Deviation.}

\item{alpha}{Value of alpha quantile used to compute portfolio VaR and CVaR; used also as quantile value for risk measures CVAR and DCVAR.}

\item{bvec}{Benchmark portfolio, a vector of length k; function \code{PortfolioOptimProjection} finds an optimal portfolio with the smallest distance to \code{bvec}.}

\item{Aconstr}{Matrix defining additional constraints,   \eqn{\dim (Aconstr) = (m,k)}, where
\eqn{k} -- number of assets, \eqn{m} -- number of constraints.}

\item{bconstr}{Vector defining additional constraints, length (\eqn{bconstr}) \eqn{ = m}.}

\item{LB}{Vector of length k, lower bounds of portfolio weights \eqn{\theta}; warning: condition LB = NULL is equivalent to LB = rep(0, k) (lower bound zero).}

\item{UB}{Vector of length k, upper bounds for portfolio weights \eqn{\theta}.}

\item{maxiter}{Maximal number of iterations.}

\item{tol}{Accuracy of computations, stopping rule.}
}
\value{
PortfolioOptimProjection returns a list with items:
\tabular{llll}{
\code{return_mean} \tab vector of asset returns  mean values. \cr

\code{mu} \tab  realized portfolio return.\cr

\code{theta} \tab  portfolio weights.\cr

\code{CVaR} \tab  portfolio CVaR.\cr

\code{VaR} \tab  portfolio VaR.\cr

\code{MAD} \tab  portfolio MAD.\cr

\code{risk} \tab  portfolio risk measured by risk measure chosen for optimization.\cr

\code{new_portfolio_return} \tab  modified target portfolio return; when the original target portfolio return \cr

\code{ } \tab is to high for the problem, the optimization problem is solved for \cr

\code{ } \tab new_portfolio_return as the target return. \cr
}
}
\description{
PortfolioOptimProjection is a linear program for financial portfolio optimization. The function finds an optimal portfolio 
which has the smallest distance to a benchmark portfolio given by \code{bvec}.
Solution is by the algorithm due to Zhao and Li modified to account for the fact that the benchmark portfolio \code{bvec} has the dimension of portfolio weights 
and the solved linear program  has a much higher dimension since the solution vector to the LP problem consists of a set of primal variables: financial portfolio weights, 
auxiliary variables coming from the reduction of the mean-risk problem to a linear program and also a set of dual variables  depending  
on the number of constrains in the primal problem (see Palczewski).
}
\examples{

library(mvtnorm)
k = 3 
num =100
dat <-  cbind(rmvnorm (n=num, mean = rep(0,k), sigma=diag(k)), matrix(1/num,num,1)) 
# a data sample with num rows and (k+1) columns for k assets;  
w_m <- rep(1/k,k) # benchmark portfolio, a vector of length k, 
port_ret = 0.05 # portfolio target return
alpha_optim = 0.95

# minimal constraints set: \\sum theta_i = 1
# has to be in two inequalities: 1 - \\epsilon <= \\sum theta_i <= 1 +\\epsilon
a0 <- rep(1,k)
Aconstr <- rbind(a0,-a0)
bconstr <- c(1+1e-8, -1+1e-8)

LB <- rep(0,k) 
UB <- rep(1,k)  

res <- PortfolioOptimProjection(dat, port_ret, risk="MAD",  
alpha=alpha_optim, w_m, Aconstr, bconstr, LB, UB, maxiter=200, tol=1e-8)

cat ( c("Projection optimal portfolio:\\n\\n"))
cat(c("weights \\n"))
print(res$theta)


cat (c ("\\n mean = ", res$mu, " risk  = ", res$risk,    "\\n CVaR = ", res$CVaR, " VaR = ",
res$VaR, "\\n MAD = ", res$MAD,  "\\n\\n"))


}
\references{
Palczewski, A., Fast LP Algorithms for Portfolio Optimization, Available at SSRN: \cr
https://papers.ssrn.com/sol3/papers.cfm?abstract_id=2951213.

Zhao, Y-B., Li, D., Locating the least 2-norm solution of linear programs via a path-following method, SIAM Journal on Optimization, 12 (2002), 893--912. DOI:10.1137/S1052623401386368.
}
