% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poped_optim.R
\name{poped_optim}
\alias{poped_optim}
\title{Optimization main module for PopED}
\usage{
poped_optim(poped.db, opt_xt = poped.db$settings$optsw[2],
  opt_a = poped.db$settings$optsw[4], opt_x = poped.db$settings$optsw[3],
  opt_samps = poped.db$settings$optsw[1],
  opt_inds = poped.db$settings$optsw[5], method = c("ARS", "BFGS", "LS"),
  control = list(), trace = TRUE,
  fim.calc.type = poped.db$settings$iFIMCalculationType,
  ofv_calc_type = poped.db$settings$ofv_calc_type,
  approx_type = poped.db$settings$iApproximationMethod,
  d_switch = poped.db$settings$d_switch,
  ED_samp_size = poped.db$settings$ED_samp_size,
  bLHS = poped.db$settings$bLHS,
  use_laplace = poped.db$settings$iEDCalculationType, out_file = "",
  parallel = F, parallel_type = NULL, num_cores = NULL,
  loop_methods = ifelse(length(method) > 1, TRUE, FALSE), iter_max = 10,
  eff_crit = 1.001, ...)
}
\arguments{
\item{poped.db}{A PopED database.}

\item{opt_xt}{Should the sample times be optimized?}

\item{opt_a}{Should the continuous design variables be optimized?}

\item{opt_x}{Should the discrete design variables be optimized?}

\item{opt_samps}{Are the nuber of sample times per group being optimized?}

\item{opt_inds}{Are the nuber of individuals per group being optimized?}

\item{method}{A vector of optimization methods to use in a sequential 
fashion.  Options are \code{c("ARS","BFGS","LS","GA")}. \code{c("ARS")} is 
for Adaptive Random Search \code{\link{optim_ARS}}.  \code{c("LS")} is for 
Line Search \code{\link{optim_LS}}. \code{c("BFGS")} is for Method 
"L-BFGS-B" from \code{\link[stats]{optim}}. \code{c("GA")} is for the 
genetic algorithm from \code{\link[GA]{ga}}.}

\item{control}{Contains control arguments for each method specified.}

\item{trace}{Should the algorithm outpur results intermittently.}

\item{fim.calc.type}{The method used for calculating the FIM. Potential values:
\itemize{
\item 0 = Full FIM.  No assumption that fixed and random effects are uncorrelated.  See \code{\link{mftot0}}.
\item 1 = Reduced FIM. Assume that there is no correlation in the FIM between the fixed and random effects, and set these elements in 
the FIM to zero. See \code{\link{mftot1}}.
\item 2 = weighted models (placeholder).
\item 3 = Not currently used.
\item 4 = Reduced FIM and computing all derivatives with respect to the standard deviation of the residual unexplained variation (sqrt(SIGMA) in NONMEM). 
This matches what is done in PFIM, and assumes that the standard deviation of the residual unexplained variation is the estimated parameter
(NOTE: NONMEM estimates the variance of the resudual unexplained variation by default). See \code{\link{mftot4}}.
\item 5 = Full FIM parameterized with A,B,C matrices & derivative of variance. See \code{\link{mftot5}}.
\item 6 = Calculate one model switch at a time, good for large matrices. See \code{\link{mftot6}}.
\item 7 = Reduced FIM parameterized with A,B,C matrices & derivative of variance See \code{\link{mftot7}}.
}}

\item{ofv_calc_type}{OFV calculation type for FIM 
\itemize{ 
\item 1 = "D-optimality". Determinant of the FIM: det(FIM)
\item 2 = "A-optimality".  Inverse of the sum of the expected parameter variances: 
1/trace_matrix(inv(FIM)) 
\item 4 = "lnD-optimality".  Natural logarithm of the determinant of the FIM: log(det(FIM)) 
\item 6 = "Ds-optimality". Ratio of the Determinant of the FIM and the Determinant of the uninteresting
rows and columns of the FIM: det(FIM)/det(FIM_u)
\item 7 = Inverse of the sum of the expected parameter RSE: 1/sum(get_rse(FIM,poped.db,use_percent=FALSE))
}}

\item{approx_type}{Approximation method for model, 0=FO, 1=FOCE, 2=FOCEI, 3=FOI.}

\item{d_switch}{\itemize{
\item \bold{******START OF CRITERION SPECIFICATION OPTIONS**********}}
D-family design (1) or ED-familty design (0) (with or without parameter uncertainty)}

\item{ED_samp_size}{Sample size for E-family sampling}

\item{bLHS}{How to sample from distributions in E-family calculations. 0=Random Sampling, 1=LatinHyperCube --}

\item{use_laplace}{Should the Laplace method be used in calculating the expectation of the OFV?}

\item{out_file}{Save output from the optimization to a file.}

\item{parallel}{Should we use parallelize the computation.}

\item{parallel_type}{Which type of parallelization should be used? 
Can be "snow" or "multicore".  "snow"  works on Linux-like systems & Windows. "multicore" works only on 
Linux-like systems.  By default this is chosen for you depending on your operating system. 
See \code{\link{start_parallel}}.}

\item{num_cores}{The number of cores to use in the parallelization.  By default  is set to the number 
 output from 
\code{parallel::detectCores()}. 
See \code{\link{start_parallel}}.}

\item{loop_methods}{Should the optimization methods be looped for
\code{iter_max} iterations, or until the efficiency of the design after the
current series (compared to the start of the series) is less than, or equal to,
\code{eff_crit}?}

\item{iter_max}{If line search is used then the algorithm tests if line
search (always run at the end of the optimization iteration) changes the 
design in any way.  If not, the algorithm stops.  If yes, then a new
iteration is run unless \code{iter_max} iterations have already been run.}

\item{eff_crit}{If \code{loop_methods==TRUE}, the looping will stop if the
efficiency of the design after the current series (compared to the start of
the series) is less than, or equal to, \code{eff_crit}.}

\item{...}{arguments passed to other functions.}
}
\description{
Optimize the objective function. The function works for both discrete and 
continuous optimization variables. If more than one optimization method is 
specified then the methods are run in series.  If \code{loop_methods=TRUE} 
then the series of optimization methods will be run for \code{iter_max} 
iterations, or until the efficiency of the design after the current series 
(compared to the start of the series) is less than \code{eff_crit}.
}
\details{
This function takes information from the PopED database supplied as an 
argument. The PopED database supplies information about the the model, 
parameters, design and methods to use. Some of the arguments coming from the 
PopED database can be overwritten; if they are supplied then they are used 
instead of the arguments from the PopED database.
}
\examples{
library(PopED)

############# START #################
## Create PopED database
## (warfarin model for optimization)
#####################################

## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

## Optimization using an additive + proportional reidual error  
## to avoid sample times at very low concentrations (time 0 or very late samples).

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.sd.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
  return(parameters) 
}

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.add.prop",
                                  bpop=c(CL=0.15, V=8, KA=1.0, Favail=1), 
                                  notfixed_bpop=c(1,1,1,0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=c(0.01,0.25),
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0,
                                  maxxt=120,
                                  a=70,
                                  mina=0,
                                  maxa=100)

############# END ###################
## Create PopED database
## (warfarin model for optimization)
#####################################


\dontrun{
  
  ##############
  # D-family Optimization
  ##############
  
  # below are a number of ways to optimize the problem
  
  
  # ARS+BFGS+LS optimization of sample times
  # optimization with just a few iterations
  # only to check that things are working
  output <- poped_optim(poped.db,opt_xt=T,
                        control = list(ARS=list(iter=5),
                                       BFGS=list(maxit=5),
                                       LS=list(line_length=5)))
  
  
  # RS+SG+LS optimization of sample times 
  # (longer run time than above but more likely to reach a maximum)
  output <- poped_optim(poped.db,opt_xt=T,parallel = TRUE)
  
  get_rse(output$FIM,output$poped.db)
  plot_model_prediction(output$poped.db)
  
  # optimization with only integer times allowed
  poped.db.2 <- poped.db
  poped.db.2$design_space$xt_space <- matrix(list(seq(1,120)),1,8)
  output_2 <- poped_optim(poped.db.2,opt_xt=T,parallel = TRUE)
  
  get_rse(output_2$FIM,output_2$poped.db)
  plot_model_prediction(output_2$poped.db)
  
  # Examine efficiency of sampling windows
  plot_efficiency_of_windows(output_2$poped.db,xt_windows=0.5)
  plot_efficiency_of_windows(output_2$poped.db,xt_windows=1)
  
  # Random search (just a few samples here)
  rs.output <- poped_optim(poped.db,opt_xt=T,method = "ARS",
                           control = list(ARS=list(iter=5)))
  
  get_rse(rs.output$FIM,rs.output$poped.db)
  
  # line search, DOSE and sample time optimization
  ls.output <- poped_optim(poped.db,opt_xt=T,opt_a=T,method = "LS",
                           control = list(LS=list(line_length=5)))
  
  # Adaptive random search, 
  # DOSE and sample time optimization
  ars.output <- poped_optim(poped.db,opt_xt=T,opt_a=T,method = "ARS",
                           control = list(ARS=list(iter=5)))
  
  # L-BFGS-B gradient search from the stats::optim() function, 
  # DOSE and sample time optimization
  bfgs.output <- poped_optim(poped.db,opt_xt=T,opt_a=T,method = "BFGS",
                            control = list(BFGS=list(maxit=5)))
  
  
  # genetic algorithm from the GA::ga() function, 
  # DOSE and sample time optimization
  ga.output <- poped_optim(poped.db,opt_xt=T,opt_a=T,method = "GA",parallel=T)
  
  ##############
  # E-family Optimization
  ##############
  
  # Adding 10\% log-normal Uncertainty to fixed effects (not Favail)
  bpop_vals <- c(CL=0.15, V=8, KA=1.0, Favail=1)
  bpop_vals_ed_ln <- cbind(ones(length(bpop_vals),1)*4, # log-normal distribution
                           bpop_vals,
                           ones(length(bpop_vals),1)*(bpop_vals*0.1)^2) # 10\% of bpop value
  bpop_vals_ed_ln["Favail",]  <- c(0,1,0)
  bpop_vals_ed_ln
  
  ## -- Define initial design  and design space
  poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                    fg_file="sfg",
                                    fError_file="feps.add.prop",
                                    bpop=bpop_vals_ed_ln, 
                                    notfixed_bpop=c(1,1,1,0),
                                    d=c(CL=0.07, V=0.02, KA=0.6), 
                                    sigma=c(0.01,0.25),
                                    groupsize=32,
                                    xt=c( 0.5,1,2,6,24,36,72,120),
                                    minxt=0,
                                    maxxt=120,
                                    a=70,
                                    mina=0,
                                    maxa=100)
  
  # ED optimization using Random search (just a few samples here)
  output <- poped_optimize(poped.db,opt_xt=1,opt_a=1,rsit=10,d_switch=0)
  get_rse(output$fmf,output$poped.db)
  
  # ED with laplace approximation, 
  # optimization using Random search (just a few samples here)
  ars.output <- poped_optim(poped.db,opt_xt=T,opt_a=T,method = "ARS",
                            d_switch=0,use_laplace=TRUE,laplace.fim=TRUE,
                            parallel=T,
                            control = list(ARS=list(iter=5)))
  
}
}
\references{
\enumerate{ \item M. Foracchia, A.C. Hooker, P. Vicini and A. 
  Ruggeri, "PopED, a software fir optimal experimental design in population 
  kinetics", Computer Methods and Programs in Biomedicine, 74, 2004. \item J.
  Nyberg, S. Ueckert, E.A. Stroemberg, S. Hennig, M.O. Karlsson and A.C. 
  Hooker, "PopED: An extended, parallelized, nonlinear mixed effects models 
  optimal design tool", Computer Methods and Programs in Biomedicine, 108, 
  2012. }
}
\seealso{
Other Optimize: \code{\link{Doptim}},
  \code{\link{LEDoptim}}, \code{\link{RS_opt_gen}},
  \code{\link{RS_opt}}, \code{\link{a_line_search}},
  \code{\link{bfgsb_min}}, \code{\link{calc_autofocus}},
  \code{\link{calc_ofv_and_grad}}, \code{\link{mfea}},
  \code{\link{optim_ARS}}, \code{\link{optim_LS}},
  \code{\link{poped_optimize}}
}

