% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smmSAR.R
\name{smmSAR}
\alias{smmSAR}
\title{Simulated Method of Moments (SMM) Estimator of SAR model}
\usage{
smmSAR(
  formula,
  contextual = FALSE,
  fixed.effects = FALSE,
  dnetwork,
  W = "identity",
  smm.ctr = list(R = 30L, iv.power = 2L, opt.tol = 1e-04, smoother = FALSE, print =
    FALSE),
  cond.var = TRUE,
  data
)
}
\arguments{
\item{formula}{object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be as for example \code{y ~ x1 + x2 | gy | gx1 + gx2}
where \code{y} is the endogenous vector, the listed variables before the pipe, \code{x1}, \code{x2} are the individual exogenous variables, \code{gy} is the average of \code{y} among friends, and
\code{gx1}, \code{gx2} are the contextual observed variables. If \code{gy} is observed and \code{gx1}, \code{gx2} are not, the formula should be
\code{y ~ x1 + x2 | gy}. If \code{gy} is not observed and \code{gx1}, \code{gx2} are, the formula should be \code{y ~ x1 + x2 || gx1 + gx2}. If \code{gy}, \code{gx1}, and \code{gx2} are not observed, the
the formula should simply be \code{y ~ x1 + x2}.}

\item{contextual}{logical; if true, this means that all individual variables will be set as contextual variables. In contrast \code{\link{mcmcSAR}},
\code{formula} as \code{y ~ x1 + x2} and \code{contextual} as \code{TRUE} is not equivalent to set formula as \code{y ~ x1 + x2 || gx1 + gx2}. \code{formula = y ~ x1 + x2} means that \code{gy}, \code{gx1}, and \code{gx2}
are not observed and \code{contextual = TRUE} means that the estimated model includes contextual effects.}

\item{fixed.effects}{logical; if true, group heterogeneity is included as fixed effects.}

\item{dnetwork}{a list, where the m-th elements is the matrix of link probability in the m-th sub-network.}

\item{W}{is the weighted-matrix in the objective function of the SMM.}

\item{smm.ctr}{is the list of some control parameters (see details).}

\item{cond.var}{logical; if true the estimator variance conditional on \code{dnetwork} will be computed.}

\item{data}{optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If missing, the variables are taken from \code{environment(formula)}, typically the environment from which \code{smmSAR} is called.}
}
\value{
A list consisting of:
\item{n.group}{number of groups.}
\item{N}{vector of each group size.}
\item{time}{elapsed time to run the SMM in second.}
\item{estimates}{vector of estimated parameters.}
\item{formula}{input value of \code{formula}.}
\item{contextual}{input value of \code{contextual}.}
\item{fixed.effects}{input value of \code{fixed.effects}.}
\item{smm.ctr}{input value of \code{smm.ctr}.}
\item{details}{other details of the model.}
}
\description{
\code{smmSAR} implements the Simulated Method of Moments (SMM) estimator of the linear-in-mean SAR model when only the linking probabilities are available or can be estimated.
}
\details{
The parameter \code{smm.ctr} is the list of some control parameters such as:
\itemize{
\item \code{R} numbers of draws R (in the package, we assume S = 1 and T = 1);
\item \code{iv.power} number of powers of the network matrix \code{G} to be used to construct instruments;
\item \code{opt.tol} optimization tolerance that will be used in \link[stats]{optimize};
\item \code{smoother} (logical) which indicates if draws should be performed using the smoother simulator;
\item \code{h} bandwith of the smoother (required if \code{smoother = TRUE});
\item \code{print} (logical) indicates if the optimization process should be printed step by step.
}
}
\examples{
\donttest{
# Number of groups
M        <- 100
# size of each group
N        <- rep(30,M)
# covariates
X        <- cbind(rnorm(sum(N),0,5),rpois(sum(N),7))
# network formation model parameter
rho      <- c(-0.8, 0.2, -0.1)
# individual effects
beta     <- c(2, 1, 1.5, 5, -3)
# endogenous effects
alpha    <- 0.4
# std-dev errors
se       <- 1
# network
tmp      <- c(0, cumsum(N))
X1l      <- lapply(1:M, function(x) X[c(tmp[x] + 1):tmp[x+1],1])
X2l      <- lapply(1:M, function(x) X[c(tmp[x] + 1):tmp[x+1],2])
dist.net <- function(x, y) abs(x - y)
X1.mat   <- lapply(1:M, function(m) {
  matrix(kronecker(X1l[[m]], X1l[[m]], FUN = dist.net), N[m])})
X2.mat   <- lapply(1:M, function(m) {
  matrix(kronecker(X2l[[m]], X2l[[m]], FUN = dist.net), N[m])})
Xnet     <- as.matrix(cbind("Const" = 1,
                            "dX1"   = mat.to.vec(X1.mat),
                            "dX2"   = mat.to.vec(X2.mat)))
ynet     <- Xnet \%*\% rho
ynet     <- c(1*((ynet + rlogis(length(ynet))) > 0))
G0       <- vec.to.mat(ynet, N, normalise = FALSE)
# normalise
G0norm   <- norm.network(G0)
# Matrix GX
GX       <- peer.avg(G0norm, X)
# simulate dependent variable use an external package
y        <- CDatanet::simsar(~ X, contextual = TRUE, Glist = G0norm,
                             theta = c(alpha, beta, se))
Gy       <- y$Gy
y        <- y$y
# build dataset
dataset           <- as.data.frame(cbind(y, X, Gy, GX))
colnames(dataset) <- c("y","X1","X2", "Gy", "GX1", "GX2")
nNet      <- nrow(Xnet) # network formation model sample size
Aobs      <- sample(1:nNet, round(0.3*nNet)) # We observed 30\%
# We can estimate rho using the gml function from the stats package
logestim  <- glm(ynet[Aobs] ~ -1 + Xnet[Aobs,], family = binomial(link = "logit"))
slogestim <- summary(logestim)
rho.est   <- logestim$coefficients
rho.var   <- slogestim$cov.unscaled # we also need the covariance of the estimator

d.logit     <- lapply(1:M, function(x) {
  out       <- 1/(1 + exp(-rho.est[1] - rho.est[2]*X1.mat[[x]] -
                            rho.est[3]*X2.mat[[x]]))
  diag(out) <- 0
  out})
smm.logit   <- smmSAR(y ~ X1 + X2, dnetwork = d.logit, contextual = TRUE,
                      smm.ctr  = list(R = 100L, print = TRUE), data = dataset)
summary(smm.logit, dnetwork = d.logit, data = dataset)
}
}
