% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/17_EVRS.R
\name{evrs}
\alias{evrs}
\title{Modelling the Economic Value of Credit Rating System}
\usage{
evrs(
  db,
  pd,
  benchmark,
  lgd,
  target,
  sigma = NA,
  r,
  elasticity,
  prob.to.leave.threshold,
  sim.num = 500,
  seed = 991
)
}
\arguments{
\item{db}{Data frame with at least the following columns: default indicator (target), PDs of model in use,
PDs of benchmark model and LGD values.}

\item{pd}{Name of PD of model in use within db argument.}

\item{benchmark}{Name of PD of benchmark model within db argument.}

\item{lgd}{Name of LGD values within db argument.}

\item{target}{Name of target (default indicator 0/1) within db argument.}

\item{sigma}{Measurement error of model in use. If default value (\code{NA}) is passed, then measurement error
is calculated as standard deviation of PD difference of model in use and benchmark model.}

\item{r}{Risk-free rate.}

\item{elasticity}{Elasticity parameter used to define customer churn in case of loan overpricing.}

\item{prob.to.leave.threshold}{Threshold for customers' probability to leave in case of loan overpricing.}

\item{sim.num}{Number of simulations. Default is 500.}

\item{seed}{Random seed to ensure reproducibility. Default is 991.}
}
\value{
The command \code{evrs} returns a list of two elements. The first element is data frame
\code{summary.tbl} and it provides simulation summary: number of simulations, number of successful simulations,
population size (number of observations of supplied \code{db} data frame), measurement error,
average churn value (number of customers that left the portfolio due to the overpricing), average return of simulated
portfolios, return of benchmark portfolio and return difference (main result of the simulation). The second element is
numeric vector of return averages of simulated portfolios.
}
\description{
\code{evrs} calculates the economic benefits of improved PD model based on increase of portfolio return.
Implemented algorithm replicates the framework presented in the Reference under assumption that
bank adopts continuous PD rating scale. Despite this assumption, results are almost identical for scenarios
of base case portfolio from the Reference.
}
\examples{
suppressMessages(library(PDtoolkit))
data(loans)
#simulate model in use
miu.formula <- Creditability ~ `Age (years)` + `Duration of Credit (month)` +
   			  `Value Savings/Stocks` + `Purpose`
miu <- glm(miu.formula, family = "binomial", data = loans)
miu.pd <- unname(predict(miu, type = "response", newdata = loans))
#simulate benchmark model with interaction.transformer support
bnm.pack <- stepFWDr(start.model = Creditability ~ 1, 
                    p.value = 0.05, 
                    db = loans,
                    check.start.model = TRUE, 
                    offset.vals = NULL)
bnm <- bnm.pack$model
bnm.pd <- unname(predict(bnm, type = "response", newdata = bnm.pack$dev.db))
#prepare data for evrs function
db <- data.frame("Creditability" = loans$Creditability, 	
   pd = miu.pd, 
   pd.benchmark = bnm.pd, 
   lgd = 0.75)
#calculate the difference in portfolio return between model in use the benchmark model
res <- evrs(db = db, 
pd = "pd", 
benchmark = "pd.benchmark", 
lgd = "lgd",
target = "Creditability",
sigma = NA, 
r = 0.03, 
elasticity = 100, 
prob.to.leave.threshold = 0.5,
sim.num = 500, 
seed = 991)
names(res)
#print simulation summary table
res[["summary.tbl"]]
#portfolio return increase in case of using benchmark model
res[["summary.tbl"]][, "return.difference", drop = FALSE]
#summary of simulated returns
summary(res[["return.sim"]])
}
\references{
Jankowitsch at al. (2007). Modelling the economic value of credit rating systems.
Journal of Banking & Finance, Volume 31, Issue 1, \doi{10.1016/j.jbankfin.2006.01.003}.
}
