% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sse.fit.r
\name{sse.fit}
\alias{sse.fit}
\title{Fit a Sum of Smooth Exponentials}
\usage{
sse.fit(data, lambda.hump = 1, lambda.sen = 10, x.hump = 35, x.sen = 50,
  kappa = 10^5, maxit = 200, verbose = FALSE)
}
\arguments{
\item{data}{data frame produced with \code{HMD2MH} or similarly structured}

\item{lambda.hump}{smoothing parameter for the hump component}

\item{lambda.sen}{smoothing parameter for the senescence part}

\item{x.hump}{upper age to compute the starting values for the hump component}

\item{x.sen}{lower age to compute the starting values for the senescence component}

\item{kappa}{ridge penalty}

\item{maxit}{maximum number of iterations of the optimization algorithm}

\item{verbose}{should the function print information during the optimization}
}
\value{
\describe{
   \item{data}{The original data frame produced with \code{HMD2MH} or similarly structured. Used later for the plot and summary methods.}
   \item{type}{The type of model used: non-parametric. To use in the summary method.}
   \item{method}{The algorithm used to fit the model: IRWLS. To use in the summary method.}
   \item{it}{The number of iterations of the IRWLS algorithm used to fit the model.}
   \item{maxit}{The maximum number of iterations of the optimization algorithm set before the estimation.}
   \item{deg2}{The degrees of freedom used for the \emph{P}-spline of the senescence component.}
   \item{deg3}{The degrees of freedom used for the \emph{P}-spline of the hump component.}
   \item{coef}{The concanated fitted coefficients of the \emph{P}-splines.}
   \item{XX}{List containing the base spline functions for each of the three components.}
   \item{mhat}{List containing three vectors containing each the age-specific mortality contributions of the estimated components. The sum of \code{mhat1}, \code{mhat2} and \code{mhat3} gives the overall fitted force of mortality.}
}
}
\description{
This function estimates a three-component non-parametric model of the age-specific mortality rates.
Its design resembles the parametric Heligman-Pollard model but it uses \emph{P}-splines instead of parametric functions to describe the force of mortality.
}
\details{
Three non-parametric components are estimated for each phase of the force of mortality.
A first component estimates the decreasing trend observed during the first years of life.
A second component estimates the concave pattern observed during adolescence and early adulthood.
A third component estimates the approximately exponential increase observed during late adulthood.

The function first estimates separate \emph{P}-splines for each components. Then, it uses a re-weighted iterative least square algorithm to adjust each component so that they sum up to the observed death rates.
The algorithm is relatively sensitive to starting values (i.e. the independent fit of each component). In case of a failed convergence, it is thus advised to reconsider the following parameters.

The \code{x.hump} and \code{x.sen} arguments are used to compute the starting values of each component. More specifically, they are used to determined the age range on which the young adult mortality hump and the senescence components are fitted.
The values of \code{x.hump} and \code{x.sen} define in a way the interval where the hump and the senescence components overlap.
In case of a very narrow hump, it is advised to reduce the value of \code{x.hump} and/or of \code{x.sen}, wherease in case of an especially wide hump it may be useful to consider larger values for \code{x.hump} and \code{x.sen}.
Inadequate values of \code{x.hump} may result in incoherent starting values for the cause-deleted SSE models and a lack of convergence.

The \code{lambda.hump} and \code{lambda.sen} parameters control the amount of smoothing imposed on the hump and senescence component respectively. Typically, the former is smaller than the latter, since this is the more instable part of the force of mortality.
However, in some cases, especially when using abridged datasets, it may be useful to consider smaller values of \code{lambda.sen}.

The \code{maxit} argument defines the maximum number of iterations used for the adjustment step. This step uses a Penalized Composite Link Model (PCLM) along with an iterative re-weighted least squares (IRWLS) optimization procedure.
The \code{maxit} argument will therefore determine the maximum number of iterations used in the IRWLS loop, but also the speed of convergence since the re-weighting defines updated solution as \deqn{new = old * (1 - it/maxit) + new * (it/maxit)}
The \code{maxit} argument is defined at 200 by default, but it can be increased in case of an absence of convergence, even if the algorithm stopped before reaching \code{maxit} number of iterations.

The \code{kappa} parameter regulates the force of the ridge penalty, which enforces the shape constraints, i.e. that the hump component is concave and the senescence component is monotonically increasing.
}
\examples{

data("CHE2010m")

fit <- sse.fit(data = CHE2010m)

}
\references{
Camarda, C. G., Eilers, P. H. C., & Gampe, J. (2016). Sums of smooth exponentials to decompose complex series of counts. Statistical Modelling.
}
\seealso{
\link{morthump}
}

