#' Run an Experiment Based on the Configuration
#'
#' Executes the experiment and saves the results to an Excel file.
#'
#' @param gptConfig A list containing the configuration for the language model, including the system prompt,
#' model specifications, and token settings.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#' Defaults to './output.xlsx' in the current working directory.
#'
#' @return This function does not return a value but saves the experiment results to the specified Excel file.
#' Upon completion, "Done." will be printed to the console.
#'
#' @export
#'
#' @examples
#' \dontrun{
#'
#' runExperiment(Experiment_config,"./output.xlsx")
#'
#' #The first argument Experiment_config is generated by preCheck() function.
#'
#' Experiment_config <- preCheck(data)
#' }
runExperiment<- function (gptConfig,savePath="./output.xlsx"){
  switch(Sys.getenv("llm"),
         "openai" = run_openai(gptConfig,savePath),
         "llama" = run_llama(gptConfig,savePath),
         "claude"= run_claude(gptConfig,savePath),
         "gemini"= run_gemini(gptConfig,savePath),
         "baichuan"= run_baichuan(gptConfig,savePath),
         "aimlapi_test"=run_openai(gptConfig,savePath),
         "custom"= run_openai(gptConfig,savePath),
         stop("Failed to the interact with the LLM.")
  )

}














#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the GPT model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the GPT model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_openai<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  systemPrompt <- gptConfig$systemPrompt
  model <- Sys.getenv("model")

  modality <-gptConfig$modality
  imgDetail <-gptConfig$imgDetail
  args <-gptConfig$args

  n <- args[["n"]]
  if (is.null(n)) {
    n <- 1
  }
  currentPrompt=""


  # csv
  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }



  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)

  # data$Event <- NA
  data$Response <- NA
  data$N <- NA
  data$Trial <- NA
  data$Message <- NA
  data$rawResponse <-NA
  data$...... <-NA

  column_names <- names(data)
  new_order <- column_names[c(6,1,2,3,9,4,5,7,8,10,11,12)]

  data <- data[, new_order]


  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]
    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      currentPrompt=""
      messages=list()
      if(systemPrompt!=""){
        messages=addMessage(messages,"system", systemPrompt)
      }

      message(r)

      for( it in unique(data$Item)){
        it_data <- r_data[r_data$Item == it, ]
        # messages=list()
        # messages=addMessage(messages,"system", systemPrompt)


      ###################################################################
      # event loopr
      for (i in seq_len(nrow(it_data))) {

        messages=addMessage(messages,"user",it_data$Prompt[i],modality,imgDetail)

        t_data <- it_data[i,]

        repeat {
          result <- tryCatch({
            if (grepl("chat", Sys.getenv("url"))) {
              result_list  <- do.call("openai_chat",modifyList(list(messages=messages, model=model),args))
            }
            else{
              result_list  <- do.call("openai_completion",modifyList(list(messages=it_data$Prompt[i], model=model),args))
            }

            content_list <- result_list$content_list
            raw_temp <- result_list$raw_response

            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(6)
          } else {
            break
          }
        }


        # if (length(content_list) == 1) {
        #   content_str <- content_list
        # } else {
        #   content_str <- paste(content_list, collapse = "\n")
        # }


        for(nr in seq_along(content_list)){{
          t_data$Response <- content_list[nr]
          t_data$N <- nr

          if (n==1){
            messages=addMessage(messages,"assistant",content_list[nr])
          }
          cMessage <- paste(messages, collapse = " ")
          if (!grepl("chat", Sys.getenv("url"))) {
            cMessage=""
          }
          new_row <- c(t_data$Session, t_data$Run,t_data$Item, t_data$Event,i,t_data$Condition, t_data$Prompt,t_data$Response, t_data$N,cMessage)

          for (i in seq(1, nchar(raw_temp), by = 25000)) {
            new_row <- c(new_row, substring(raw_temp, i, min(i + 24999, nchar(raw_temp))))
          }
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

        }
          saveWorkbook(wb, savePath, overwrite = TRUE)
        }
      }
    }
    }
    ###################################################################
  }


  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)

    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)
    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}







#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the llama model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the llama model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_llama<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  systemPrompt <- gptConfig$systemPrompt
  args <-gptConfig$args
  n <- 1

  #csv
  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }

  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)


  column_names <- names(data)
  new_order <- column_names[c(6,1,2,3,9,4,5,7,8,10,11,12)]

  data <- data[, new_order]

  data$Response <- NA
  data$N <- NA
  data$Trial <- NA
  data$Message <- NA
  data$raw_Response <-NA
  data$...... <-NA
  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]

    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      messages=""
      if(systemPrompt!=""){
        messages=addMessage(messages,"system", systemPrompt)
      }
      

      message(r)
      for( it in unique(data$Item)){
        it_data <- r_data[r_data$Item == it, ]
        # messages=""
        # messages=addMessage(messages,"system", systemPrompt)
        # event loopr

      #####TODO:2222
      # Trial loop
      for (i in seq_len(nrow(it_data))) {
        messages=addMessage(messages,"user",it_data$Prompt[i])
        t_data <- it_data[i,]

        repeat {
          result <- tryCatch({
            # TODO:111
            result_list  <- do.call("llama_chat",modifyList(list(messages=messages),args))
            content_list <- result_list$content_list
            raw_temp <- result_list$raw_response
            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(5)
          } else {
            break
          }
        }

          t_data$Response <- content_list
          t_data$N <- 1


          messages=addMessage(messages,"assistant",content_list)


          cMessage <- paste(messages, collapse = " ")

          # new_row <- c(t_data$Run, t_data$Item,t_data$Event, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          new_row <- c(t_data$Session, t_data$Run,t_data$Item, t_data$Event,i,t_data$Condition, t_data$Prompt,t_data$Response, t_data$N,cMessage)
          for (i in seq(1, nchar(raw_temp), by = 25000)) {
           new_row <- c(new_row, substring(raw_temp, i, min(i + 24999, nchar(raw_temp))))
          }

        writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

          saveWorkbook(wb, savePath, overwrite = TRUE)


      }
    }
  }
  }
  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)
    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)

    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}



#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the llama model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the llama model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_claude<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  version <- gptConfig$version
  systemPrompt <- gptConfig$systemPrompt
  model <- gptConfig$model
  args <-gptConfig$args
  n <- 1
  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }





  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)




  data$Response <- NA
  data$N <- NA
  data$Trial <- NA
  data$Message <- NA
  data$rawResponse <-NA
  data$...... <-NA

  column_names <- names(data)
  new_order <- column_names[c(6,1,2,3,9,4,5,7,8,10,11,12)]

  data <- data[, new_order]

  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]
    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      messages=list()

      message(r)

      for( it in unique(data$Item)){
        it_data <- r_data[r_data$Item == it, ]
        # messages=list()

      # Trial loop
        for (i in seq_len(nrow(it_data))) {

        messages=addMessage(messages,"user",it_data$Prompt[i],modality="base")
        t_data <- it_data[i,]

        repeat {
          result <- tryCatch({
            #content_list = claude_chat(version=version,messages=messages, system = systemPrompt, model=model, temperature = temperature,max_tokens =  max_tokens, stop_sequences = stop_sequences)
            result_list  <- do.call("claude_chat",modifyList(list(messages=messages, model=model,version=version,system=systemPrompt),args))
            content_list <- result_list$content_list
            raw_temp <- result_list$raw_response

            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(6)
          } else {
            break
          }
        }


        if (length(content_list) == 1) {
          content_str <- content_list[[1]]
        } else {
          content_str <- paste(content_list, collapse = "\n")
        }


        # N responses loop
        for(nr in seq_along(content_list)){{
          t_data$Response <- content_list[nr]
          t_data$N <- nr

          if (n==1){
            messages=addMessage(messages,"assistant",content_str,modality = "base")

          }
          cMessage <- paste(messages, collapse = " ")


          # new_row <- c(t_data$Run, t_data$Item,t_data$Event, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          new_row <- c(t_data$Session, t_data$Run,t_data$Item, t_data$Event,i,t_data$Condition, t_data$Prompt,t_data$Response, t_data$N,cMessage)
          for (i in seq(1, nchar(raw_temp), by = 25000)) {
            new_row <- c(new_row, substring(raw_temp, i, min(i + 24999, nchar(raw_temp))))
          }
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

        }
          saveWorkbook(wb, savePath, overwrite = TRUE)
        }
      }
      }
    }
  }

  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)

    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)
    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}



#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the llama model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the llama model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_gemini<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  model <- Sys.getenv("model")
  args <-gptConfig$args
  n <- 1
  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }





  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)




  data$Response <- NA
  data$N <- NA
  data$Trial <- NA
  data$Message <- NA
  data$rawResponse <-NA
  data$...... <-NA

  column_names <- names(data)
  new_order <- column_names[c(6,1,2,3,9,4,5,7,8,10,11,12)]
 
  data <- data[, new_order]


  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]
    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      messages=list()

      message(r)
      # Trial loop
      for( it in unique(data$Item)){
        it_data <- r_data[r_data$Item == it, ]
        # messages=list()

      for (i in seq_len(nrow(it_data))) {

        messages=addMessage(messages,"user",it_data$Prompt[i],modality="base")
        t_data <- it_data[i,]

        repeat {
          result <- tryCatch({
            #content_list = gemini_chat(messages=messages,model=model, temperature = temperature,max_tokens =  max_tokens, stop_sequences = stop_sequences,topK=topK,topP=topP)
            result_list  <- do.call("gemini_chat",modifyList(list(messages=messages, model=model),args))
            content_list <- result_list$content_list
            raw_temp <- result_list$raw_response
            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(6)
          } else {
            break
          }
        }


        if (length(content_list) == 1) {
          content_str <- content_list[[1]]
        } else {
          content_str <- paste(content_list, collapse = "\n")
        }


        # N responses loop
        for(nr in seq_along(content_list)){{
          t_data$Response <- content_list[nr]
          t_data$N <- nr

          if (n==1){
            messages=addMessage(messages,"model",content_str,modality = "base")

          }
          cMessage <- paste(messages, collapse = " ")

          # new_row <- c(t_data$Run, t_data$Item,t_data$Event, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          new_row <- c(t_data$Session, t_data$Run,t_data$Item, t_data$Event,i,t_data$Condition, t_data$Prompt,t_data$Response, t_data$N,cMessage)
          for (i in seq(1, nchar(raw_temp), by = 25000)) {
            new_row <- c(new_row, substring(raw_temp, i, min(i + 24999, nchar(raw_temp))))
          }
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

        }
          saveWorkbook(wb, savePath, overwrite = TRUE)
        }
      }
    }
  }
  }
  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)

    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)
    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}


#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the llama model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the llama model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_baichuan<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  model <- gptConfig$model
  args <-gptConfig$args
  n <- args[["n"]]
  if (is.null(n)) {
    n <- 1
  }
  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }





  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)




  data$Response <- NA

  data$N <- NA
  data$Trial <- NA
  data$Message <- NA
  data$rawResponse <-NA
  data$...... <-NA

  column_names <- names(data)
  new_order <- column_names[c(6,1,2,3,9,4,5,7,8,10,11,12)]

  data <- data[, new_order]

  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]
    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      messages=list()
      message(r)

      for( it in unique(data$Item)){
        it_data <- r_data[r_data$Item == it, ]
        # messages=list()

      # Trial loop
      for (i in seq_len(nrow(it_data))) {

        messages=addMessage(messages,"user",it_data$Prompt[i])
        t_data <- it_data[i,]

        repeat {
          result <- tryCatch({
            #content_list = baichuan_chat(messages = messages , model = model, temperature = temperature, max_tokens = max_tokens, top_k = top_k ,top_p = top_p)
            result_list  <- do.call("baichuan_chat",modifyList(list(messages=messages, model=model),args))
            content_list <- result_list$content_list
            raw_temp <- result_list$raw_response

            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(6)
          } else {
            break
          }
        }


        if (length(content_list) == 1) {
          content_str <- content_list
        } else {
          content_str <- paste(content_list, collapse = "\n")
        }


        # N responses loop
        for(nr in seq_along(content_list)){{
          t_data$Response <- content_list[nr]
          t_data$N <- nr

          if (n==1){
            messages=addMessage(messages,"assistant",content_list[nr])

          }
          cMessage <- paste(messages, collapse = " ")

          # new_row <- c(t_data$Run, t_data$Item,t_data$Event, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          new_row <- c(t_data$Session, t_data$Run,t_data$Item, t_data$Event,i,t_data$Condition, t_data$Prompt,t_data$Response, t_data$N,cMessage)
          for (i in seq(1, nchar(raw_temp), by = 25000)) {
            new_row <- c(new_row, substring(raw_temp, i, min(i + 24999, nchar(raw_temp))))
          }
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

        }
          saveWorkbook(wb, savePath, overwrite = TRUE)
        }
      }
    }
  }
  }
  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)

    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)
    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}

