#' @title Maximum independent features
#' @description The maximum number of independent features in principle from
#' all the haplotypes generated by Haplotype Trend Regression with eXtra flexibility (HTRX) (\code{\link{htrx_max}}) or Haplotype Trend Regression (HTR) (\code{\link{htr_max}}).
#' @name htrx_max
#' @param nsnp a positive integer giving the number of single nucleotide polymorphisms (SNPs) included in the haplotypes.
#' @param n_haps a positive integer giving the number of haplotypes,
#' which is also the number of columns of the HTRX or HTR matrix.
#' By default, n_haps=3^nsnp-1
#' @param cap a positive integer which manually sets the maximum number of independent features.
#' By default, cap=40.
#'
#' @details The maximum number of independent features in principle is \ifelse{html}{\out{2<sup>nsnp</sup>}}{\eqn{2^nsnp}}-1.
#' You can also manually set the upper limit of independent features (by setting "cap") that can be included in the final HTRX or HTR model.
#'
#' @return a positive integer giving the maximum number of independent features to be included in the analysis.
#' @examples
#' ## the maximum number of independent haplotypes consisted of 4 SNPs from HTRX
#' htrx_max(nsnp=4,n_haps=(3^4-1))
#'
#' ## the maximum number of independent haplotypes consisted of 6 SNPs from HTR
#' htr_max(nsnp=6)
NULL

#' @rdname htrx_max
#' @export
htrx_max<-function(nsnp,n_haps=3^nsnp-1,cap=40){
  ## This is based on the theoretical number of linearly independent features:
  ## If we have N snps, there are at most 2^n-1 independent haplotypes
  n_hapx=2^nsnp-1 ## Dimension of HTR
  if(n_hapx>n_haps) n_hapx=n_haps;
  if(n_hapx>cap) n_hapx=cap;
  return(n_hapx)
}

#' @rdname htrx_max
#' @export
htr_max<-function(nsnp,cap=40){
  ## This is based on the theoretical number of linearly independent features:
  ## If we have N snps, there are at most 2^n-1 independent haplotypes
  n_hap=2^nsnp-1
  if(n_hap>cap) n_hap=cap;
  return(n_hap)
}
