
#' Approximate GP simulation
#'
#' Calculates an approximation to the inverse Cholesky
#' factor of the covariance matrix using Vecchia's approximation,
#' then the simulation is produced by solving a linear system
#' with a vector of uncorrelated standard normals
#' @param m Number of nearest neighbors to use in approximation
#' @inheritParams vecchia_loglik
#' @return vector of simulated values
#' @examples
#' locs <- as.matrix( expand.grid( (1:100)/100, (1:100)/100 ) )
#' y <- fast_Gp_sim(c(4,0.2,0.5,0), "matern_isotropic",  locs, 30 )
#' fields::image.plot( matrix(y,100,100) )
#' @export
fast_Gp_sim <- function( covparms, covfun_name = "matern_isotropic", locs, m = 30 ){
    
    # figure out if lonlat or not
    if( covfun_name == "matern_sphere" || covfun_name == "matern_sphere_time" ){
        lonlat <- TRUE
    } else {
        lonlat <- FALSE
    }
    
    if( covfun_name == "matern_space_time" ){
        space_time <- TRUE
    } else {
        space_time <- FALSE
    }

    n <- nrow(locs)
    m <- min(m,n-1)
    ord <- order_maxmin(locs,lonlat=lonlat,space_time=space_time)
    locsord <- locs[ord,]
    NNarray <- find_ordered_nn(locsord,m,lonlat=lonlat,space_time=space_time)
    Linv <- vecchia_Linv( covparms, covfun_name, locsord, NNarray )
    y <- fast_Gp_sim_Linv( Linv, NNarray )
    y[ord] <- y
    return(y)

}


#' Approximate GP simulation with specified Linverse
#'
#' In situations where we want to do many gaussian process
#' simulations from the same model, we can compute Linverse
#' once and reuse it, rather than recomputing for each identical simulation.
#' This function also allows the user to input the vector of standard normals \code{z}.
#' @param Linv Matrix containing the entries of Linverse, usually the output from
#' \code{vecchia_Linv}.
#' @param NNarray Matrix of nearest neighbor indices, usually the output from \code{\link{find_ordered_nn}}
#' @param z Optional vector of standard normals. If not specified,
#' these are computed within the function.
#' @return vector of simulated values
#' @examples
#' locs <- as.matrix( expand.grid( (1:100)/100, (1:100)/100 ) )
#' ord <- order_maxmin(locs)
#' locsord <- locs[ord,]
#' m <- 10
#' NNarray <- find_ordered_nn(locsord,m)
#' covparms <- c(2, 0.2, 1, 0)
#' Linv <- vecchia_Linv( covparms, "matern_isotropic", locsord, NNarray )
#' y <- fast_Gp_sim_Linv(Linv,NNarray)
#' y[ord] <- y
#' fields::image.plot( matrix(y,100,100) )
#' @export
fast_Gp_sim_Linv <- function( Linv, NNarray, z = NULL ){

    if( is.null(z) ){ z = stats::rnorm(nrow(Linv)) }
    y <- L_mult( Linv, z, NNarray )
    return(y)

}
