% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/easyGParetoptim.R
\name{easyGParetoptim}
\alias{easyGParetoptim}
\title{EGO algorithm for multiobjective optimization}
\usage{
easyGParetoptim(fn, cheapfn = NULL, budget, lower, upper, par = NULL,
  value = NULL, control = list(method = "SMS", trace = 1, inneroptim =
  "pso", maxit = 100, seed = 42), ...)
}
\arguments{
\item{fn}{the multi-objective function to be minimized (vectorial output), found by a call to \code{\link[base]{match.fun}},}

\item{cheapfn}{optional additional fast-to-evaluate objective function (handled next with class \code{\link[GPareto]{fastfun}}),
which does not need a kriging model, handled by a call to \code{\link[base]{match.fun}},}

\item{budget}{total number of calls to the objective function,}

\item{lower}{vector of lower bounds for the variables to be optimized over,}

\item{upper}{vector of upper bounds for the variables to be optimized over,}

\item{par}{initial design of experiments. If not provided, \code{par} is taken as a maximin LHD with budget/3 points,}

\item{value}{initial set of objective observations \eqn{f(par)}. Computed if not provided.
Not that \code{value} may NOT contain any \code{cheapfn} value,}

\item{control}{an optional list of control parameters. See "Details",}

\item{...}{additional parameters to be given to the objective \code{fn}.}
}
\value{
A list with components:
\itemize{
\item{\code{par}}{: all the non-dominated points found,}
\item{\code{value}}{: the matrix of objective values at the points given in \code{par},}
\item{\code{history}}{: a list containing all the points visited by the algorithm (\code{X}) and their corresponding objectives (\code{y}),}
\item{\code{model}}{: a list of objects of class \code{\link[DiceKriging]{km}}, corresponding to the last kriging models fitted.}
}
}
\description{
User-friendly wrapper of the function \code{\link[GPareto]{GParetoptim}}.
Generates initial DOEs and kriging models (objects of class \code{\link[DiceKriging]{km}}),
and executes \code{nsteps} iterations of multiobjective EGO methods.
}
\details{
Does not require specific knowledge on kriging models (objects of class \code{\link[DiceKriging]{km}}).\cr

The problem considered is of the form: \eqn{min f(x) = f_1(x), ..., f_p(x)}.
The \code{control} argument is a list that can supply any of the following optional components: \cr
\itemize{
\item \code{method}{: choice of multiobjective improvement function: "\code{SMS}", "\code{EHI}", "\code{EMI}" or "\code{SUR}"
(see \code{\link[GPareto]{crit_SMS}}, \code{\link[GPareto]{crit_EHI}}, \code{\link[GPareto]{crit_EMI}}, \code{\link[GPareto]{crit_SUR}}),}
\item \code{trace}{:  if positive, tracing information on the progress of the optimization is produced,}
\item \code{inneroptim}{: choice of the inner optimization algorithm: "\code{genoud}", "\code{pso}" or "\code{random}"
 (see \code{\link[rgenoud]{genoud}} and \code{\link[pso]{psoptim}}),}
\item \code{maxit}{: maximum number of iterations of the inner loop,}
\item \code{seed}{: to fix the random variable generator,}
\item \code{refPoint}{: reference point for hypervolume computations (for "\code{SMS}" and "\code{EHI}" methods).}
}
For additional details or other possible arguments, see \code{\link[GPareto]{GParetoptim}}.\cr

Display of results and various post-processings are available with \code{\link[GPareto]{plotGPareto}}.
}
\examples{
#---------------------------------------------------------------------------
# 2D objective function, 3 cases
#---------------------------------------------------------------------------
\dontrun{
set.seed(25468)
n_var <- 2
fname <- ZDT3
lower <- rep(0, n_var)
upper <- rep(1, n_var)

#---------------------------------------------------------------------------
# 1- Expected Hypervolume Improvement optimization, using pso
#---------------------------------------------------------------------------
res <- easyGParetoptim(fn=fname, lower=lower, upper=upper, budget=15,
                   control=list(method="EHI", inneroptim="pso", maxit=20))
par(mfrow=c(1,2))
plotGPareto(res)
title("Pareto Front")
plot(res$history$X, main="Pareto set", col = "red", pch = 20)
points(res$par, col="blue", pch = 17)

#---------------------------------------------------------------------------
# 2- SMS Improvement optimization using random search, with initial DOE given
#---------------------------------------------------------------------------
library(DiceDesign)
design.init   <- maximinESE_LHS(lhsDesign(10, n_var, seed = 42)$design)$design
response.init <- t(apply(design.init, 1, fname))
res <- easyGParetoptim(fn=fname, par=design.init, value=response.init, lower=lower, upper=upper,
                       budget=15, control=list(method="SMS", inneroptim="random", maxit=100))
par(mfrow=c(1,2))
plotGPareto(res)
title("Pareto Front")
plot(res$history$X, main="Pareto set", col = "red", pch = 20)
points(res$par, col="blue", pch = 17)

#---------------------------------------------------------------------------
# 3- Stepwise Uncertainty Reduction optimization, with one fast objective function
#---------------------------------------------------------------------------
fname <- camelback
cheapfn <- function(x) {
if (is.null(dim(x))) return(-sum(x))
else return(-rowSums(x))
}
res <- easyGParetoptim(fn=fname, cheapfn=cheapfn, lower=lower, upper=upper, budget=15,
                   control=list(method="SUR", inneroptim="pso", maxit=20))
par(mfrow=c(1,2))
plotGPareto(res)
title("Pareto Front")
plot(res$history$X, main="Pareto set", col = "red", pch = 20)
points(res$par, col="blue", pch = 17)
}
}
\author{
Victor Picheny (INRA, Castanet-Tolosan, France)

Mickael Binois (Mines Saint-Etienne/Renault, France)
}
\references{
M. T. Emmerich, A. H. Deutz, J. W. Klinkenberg (2011), Hypervolume-based expected improvement: Monotonicity properties and exact computation,
\emph{Evolutionary Computation (CEC)}, 2147-2154. \cr \cr
V. Picheny (2015), Multiobjective optimization using Gaussian process emulators via stepwise uncertainty reduction,
\emph{Statistics and Computing}, 25(6), 1265-1280. \cr \cr
T. Wagner, M. Emmerich, A. Deutz, W. Ponweiser (2010), On expected-improvement criteria for model-based multi-objective optimization.
\emph{Parallel Problem Solving from Nature}, 718-727, Springer, Berlin. \cr \cr
J. D. Svenson (2011), \emph{Computer Experiments: Multiobjective Optimization and Sensitivity Analysis}, Ohio State university, PhD thesis.
}

