% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_game.R
\name{solve_game}
\alias{solve_game}
\title{Main solver}
\usage{
solve_game(fun, ..., equilibrium = "NE", crit = "sur", model = NULL,
  n.init = NULL, n.ite, d, nobj, x.to.obj = NULL, noise.var = NULL,
  integcontrol = NULL, simucontrol = NULL, plotcontrol = NULL,
  filtercontrol = NULL, kmcontrol = NULL, returncontrol = NULL,
  ncores = 1, trace = 1, seed = NULL)
}
\arguments{
\item{fun}{fonction with vectorial output}

\item{...}{additional parameter to be passed to \code{fun}}

\item{equilibrium}{either '\code{NE}', '\code{KSE}' or '\code{NKSE}' for Nash/Kalai-Smoridinsky/Nash-Kalai-Smoridinsky equilibria}

\item{crit}{'\code{sur}' (default) is available for all equilibria, '\code{psim}' and '\code{pex}' are available for Nash}

\item{model}{list of \code{\link[DiceKriging]{km}} models}

\item{n.init}{number of points of the initial design of experiments if no model is given}

\item{n.ite}{number of iterations of sequential optimization}

\item{d}{variable dimension}

\item{nobj}{number of objectives (players)}

\item{x.to.obj}{for \code{NE} and \code{NKSE}, which variables for which objective}

\item{noise.var}{noise variance. Either a scalar (same noise for all objectives), a vector (constant noise, different for each objective),
a function (type closure) with vectorial output (variable noise, different for each objective) or \code{"given_by_fn"}, see Details.
If not provided, \code{noise.var} is taken as the average of \code{model@noise.var}.}

\item{integcontrol}{optional list for handling integration points. See Details.}

\item{simucontrol}{optional list for handling conditional simulations. See Details.}

\item{plotcontrol}{optional list for handling during-optimization plots. See Details.}

\item{filtercontrol}{optional list for handling filters. See Details.}

\item{kmcontrol}{optional list for handling \code{\link[DiceKriging]{km}} models. See Details.}

\item{returncontrol}{optional list for choosing return options. See Details.}

\item{ncores}{number of CPU available (> 1 makes mean parallel \code{TRUE})}

\item{trace}{controls the level of printing: \code{0} (no printing), \code{1} (minimal printing), \code{3} (detailed printing)}

\item{seed}{to fix the random variable generator}
}
\value{
A list with components:
\itemize{
\item{\code{model}}{: a list of objects of class \code{\link[DiceKriging]{km}} corresponding to the last kriging models fitted.}
\item{\code{Jplus}}{: recorded values of the acquisition function maximizer}
\item{\code{integ.pts} and  \code{expanded.indices}}{: the discrete space used,}
\item{\code{predEq}}{: a list containing the recorded values of the estimated best solution,}
\item{\code{Eq.design, Eq.poff}}{: estimated equilibrium and corresponding pay-off (if \code{return.Eq==TRUE})}
}
}
\description{
Main function to solve games.
}
\details{
If \code{noise.var="given_by_fn"}, \code{fn} returns a list of two vectors, the first being the objective functions and the second
the corresponding noise variances.

\code{integcontrol} controls the way the design space is discretized. One can directly provide a set of points \code{integ.pts} with
corresponding indices \code{expanded.indices} (for \code{NE}). Otherwise, the points are generated according to the number of strategies \code{n.s}.
If \code{n.s} is a scalar, it corresponds to the total number of strategies (to be divided equally among players),
otherwise it corresponds to the nb of strategies per player. In addition, one may choose the type of discretization with \code{gridtype}.
Options are '\code{lhs}' or '\code{cartesian}'. Finally, \code{lb} and \code{ub} are vectors specifying the bounds for the design variables.
By default the design space is \code{[0,1]^d}.

\code{simucontrol} controls options on conditional GP simulations. Options are \code{IS}: if \code{TRUE}, importance sampling is used for \code{ynew};
\code{n.ynew} number of samples of \code{Y(x_{n+1})} and \code{n.sim} number of sample path generated.

\code{plotcontrol} can be used to generate plots during the search. Options are \code{plots} (Boolean, \code{FALSE} by default), \code{compute.actual}
(Boolean, \code{FALSE} by default, to draw the actual problem, only for inexpensive \code{fun}), and \code{pbname} (string, for figure title and pdf export).

\code{filtercontrol} controls filtering options. \code{filter} sets how to select a subset of simulation and candidate points,
either either a single value or a vector of two to use different filters for simulation and candidate points.
Possible values are '\code{window}', '\code{Pnash}' (for \code{NE}), '\code{PND}' (probability of non domination), '\code{none}'.
\code{nsimPoints} and \code{ncandPoints} set the maximum number of simulation/candidate points wanted
(use with filter '\code{Pnash}' for now). Default values are \code{800} and \code{200}, resp.
\code{randomFilter} (\code{TRUE} by default) sets whereas the filter acts randomly or deterministically.

\code{kmcontrol} Options for handling \code{nobj} \code{\link[DiceKriging]{km}} models.
\code{cov.reestim} (Boolean, \code{TRUE} by default) specifies if the kriging hyperparameters
should be re-estimated at each iteration,

\code{returncontrol} sets options for the last iterations and what is returned by the algorithm.
\code{return.Eq} (Boolean, \code{TRUE} by default) specifies
if a final search for the equilibrium is performed at the end. \code{finalcrit} sets a different criterion for the last iteration.
\code{track.Eq} allows to estimate the equilibrium at each iteration; options are '\code{none}' to do nothing,
"\code{mean}" (default) to compute the equilibrium of the prediction mean (all candidates),
 "\code{empirical}" (for \code{KSE}) and "\code{pex}"/"\code{psim}" (\code{NE} only)
for using \code{Pnash} estimate (along with mean estimate, on integ.pts only, NOT reestimated if \code{filter.simu} or \code{crit} is \code{Pnash}).
}
\examples{
\dontrun{

##############################################
# Example 1: 2 variables, 2 players, no filter
##############################################
# Define objective function (R^2 -> R^2)
fun <- function (x)
{
  if (is.null(dim(x)))    x <- matrix(x, nrow = 1)
  b1 <- 15 * x[, 1] - 5
  b2 <- 15 * x[, 2]
  return(cbind((b2 - 5.1*(b1/(2*pi))^2 + 5/pi*b1 - 6)^2 + 10*((1 - 1/(8*pi)) * cos(b1) + 1),
               -sqrt((10.5 - b1)*(b1 + 5.5)*(b2 + 0.5)) - 1/30*(b2 - 5.1*(b1/(2*pi))^2 - 6)^2-
                1/3 * ((1 - 1/(8 * pi)) * cos(b1) + 1)))
}

# To use parallel computation (turn off on Windows)
library(parallel)
parallel <- FALSE #TRUE # 
if(parallel) ncores <- detectCores() else ncores <- 1

# Simple configuration: no filter, discretization is a 21x21 grid

# Grid definition
n.s <- rep(21, 2)
x.to.obj   <- c(1,2)
gridtype <- 'cartesian'

# Run solver with 6 initial points, 4 iterations
# Increase n.ite to at least 10 for better results
res <- solve_game(fun, equilibrium = "NE", crit = "sur", n.init=6, n.ite=4,
                  d = 2, nobj=2, x.to.obj = x.to.obj,
                  integcontrol=list(n.s=n.s, gridtype=gridtype),
                  ncores = ncores, trace=1, seed=1)

# Get estimated equilibrium and corresponding pay-off
NE <- res$Eq.design
Poff <- res$Eq.poff

# Draw results
plotGame(res)

##############################################
# Example 2: 4 variables, 2 players, filtering
##############################################
fun <- function(x, nobj = 2){
  if (is.null(dim(x)))     x <- matrix(x, 1)
  y <- matrix(x[, 1:(nobj - 1)], nrow(x))
  z <- matrix(x[, nobj:ncol(x)], nrow(x))
  g <- rowSums((z - 0.5)^2)
  tmp <- t(apply(cos(y * pi/2), 1, cumprod))
  tmp <- cbind(t(apply(tmp, 1, rev)), 1)
  tmp2 <- cbind(1, t(apply(sin(y * pi/2), 1, rev)))
  return(tmp * tmp2 * (1 + g))
}

# Grid definition: player 1 plays x1 and x2, player 2 x3 and x4
# The grid is a lattice made of two LHS designs of different sizes
n.s <- c(44, 43)
x.to.obj   <- c(1,1,2,2)
gridtype <- 'lhs'

# Set filtercontrol: window filter applied for integration and candidate points
# 500 simulation and 200 candidate points are retained.
filtercontrol <- list(nsimPoints=500, ncandPoints=200,
                   filter=c("window", "window"))

# Set km control: lower bound is specified for the covariance range
# Covariance type and model trend are specified
kmcontrol <- list(lb=rep(.2,4), model.trend=~1, covtype="matern3_2")

# Run solver with 20 initial points, 4 iterations
# Increase n.ite to at least 20 for better results
res <- solve_game(fun, equilibrium = "NE", crit = "psim", n.init=20, n.ite=2,
                  d = 4, nobj=2, x.to.obj = x.to.obj,
                  integcontrol=list(n.s=n.s, gridtype=gridtype),
                  filtercontrol=filtercontrol,
                  kmcontrol=kmcontrol,
                  ncores = 1, trace=1, seed=1)

# Get estimated equilibrium and corresponding pay-off
NE <- res$Eq.design
Poff <- res$Eq.poff

# Draw results
plotGame(res)

}
}
\references{
V. Picheny, M. Binois, A. Habbal (2016+), A Bayesian optimization approach to find Nash equilibria,
\emph{https://arxiv.org/abs/1611.02440}.
}

