#' Create a simple design with spillovers
#'
#' Builds a design with \code{N_groups} groups each containing \code{N_i_group} individuals. 
#' Potential outcomes exhibit spillovers: if any individual in a group receives treatment, 
#' the effect is spread equally among members of the group. 
#' 
#' @details
#' 
#' Parameter \code{gamma} controls interactions between spillover effects.For \code{gamma}=1 for every $1 given to a member of a group, each member receives $1*\code{N_i_group} no matter how many others are already treated.
#' For \code{gamma}>1 (<1) for every $1 given to a member of a group, each member receives an amount that depends negatively (positively) on the number already treated.  
#' 
#' The default estimand is the average difference across subjects between no one treated and only that subject treated.  
#' 
#' See \href{https://declaredesign.org/library/articles/simple_spillover.html}{vignette online}.
#' 
#' @param N_groups An integer. Number of groups.
#' @param N_i_group Number of units in each group. Can be scalar or vector of length \code{N_groups}.
#' @param sd_i A nonnegative number. Standard deviation of individual-level shock.
#' @param gamma A number. Parameter that controls whether spillovers within groups substitute or complement each other. See `Details`.
#' @return A simple spillover design.
#' @author \href{https://declaredesign.org/}{DeclareDesign Team}
#' @concept experiment
#' @concept spillovers
#' @importFrom DeclareDesign declare_assignment declare_estimand declare_estimator declare_population declare_reveal
#' @importFrom fabricatr fabricate add_level fabricate
#' @importFrom randomizr conduct_ra 
#' @importFrom estimatr tidy lm_robust
#' @export
#' @examples
#' # Generate a simple spillover design using default arguments:
#' simple_spillover_design <- simple_spillover_designer()
#'


simple_spillover_designer <- function(N_groups = 80, 
                                      N_i_group = 3, 
                                      sd_i = .2,
                                      gamma = 2)
{
  if(sd_i < 0) stop("sd_i must be nonnegative")
  if(N_i_group < 1 || N_groups < 1) stop("N_i_group and N_groups must be equal to or greater than 1")
  {{{
    # M: Model
    population <- declare_population(G = add_level(N = N_groups, n = N_i_group), 
                              i = add_level(N = n, zeros = 0, ones = 1))
    
    dgp <- function(i, Z, G, n) (sum(Z[G == G[i]])/n[i])^gamma + rnorm(1)*sd_i
    
    # I: Inquiry
    estimand <- declare_estimand(Treat_1 = mean(
      sapply(1:length(G), function(i) {
        Z_i <- (1:length(G)) == i
        dgp(i,Z_i,G, n) - dgp(i, zeros, G, n)})
    ), label = "estimand")
    
    # D: Data Strategy
    assignment <- declare_assignment()
    
    reveal_Y <- declare_reveal(handler=fabricate,
                             Y = sapply(1:N, function(i) dgp(i, Z, G, n)))
    
    # A: Answer Strategy
    estimator <- declare_estimator(Y ~ Z, estimand = estimand, 
                                   model = lm_robust, label = "naive")
    
    # Design
    simple_spillover_design <- population + estimand + assignment + reveal_Y + estimator
    
  }}}
  attr(simple_spillover_design, "code") <- 
    construct_design_code(simple_spillover_designer, match.call.defaults())
  
  simple_spillover_design
}

attr(simple_spillover_designer, "shiny_arguments") <- list(
  N_groups = c(50, 100, 500),
  N_i_group = c(10, 50, 100),
  sd_i = c(0, .5, 1),
  gamma = c(-2, 2)
)

attr(simple_spillover_designer, "tips") <-
  list(
    N_groups = "Number of groups",
    N_i_group = "Number of units in each group",
    sd_i = "Standard deviation of individual-level shock",
    gamma = "Parameter that controls whether spillovers within groups substitute or complement each other"
  )

attr(simple_spillover_designer, "description") <- "
<p> A spillover design with <code>N_groups</code> groups each containing 
<code>N_i_group</code> individuals. Potential outcomes exhibit spillovers: if 
any individual in a group receives treatment, the effect is spread equally among 
members of the group."
