% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fastGPfit.R
\name{GP.Bayes.fit}
\alias{GP.Bayes.fit}
\title{Regular Bayesian fitting of Gaussian process regression on regular grid points with the modified exponential sqaured kernel.}
\usage{
GP.Bayes.fit(
  y,
  x,
  poly_degree = 60,
  a = 0.01,
  b = 20,
  num_results = 500L,
  iters_between_results = 2L,
  burn_in = 500L,
  a_sigma = 0.01,
  b_sigma = 0.01,
  a_zeta = 0.01,
  b_zeta = 0.01,
  center = NULL,
  scale = NULL,
  max_range = NULL,
  progress_bar = FALSE
)
}
\arguments{
\item{y}{A vector of real numbers as the observations for the reponse variable.}

\item{x}{A matrix of real numbers as grid points where rows are observations and columns are coordinates.}

\item{poly_degree}{A integer number to specify the highest degree of Hermite polynomials. The default value is 10L.}

\item{a}{A positive real number to specify the concentration parameter in the standard modified exponential squared kernel. The larger value the more the GP concentrates around the center. The default value is 0.01.}

\item{b}{A positive real number to specify the smoothness parameter in the standard modified exponential squared kernel. The smaller value the smoother the GP is. The default value is 1.0.}

\item{num_results}{An integer number to specify the number of posterior samples to save over MCMC iterations.}

\item{iters_between_results}{An integer number to specify the number of iterations to skip between two saved iterations.}

\item{burn_in}{An integer number to specify the burn-in number. The default value is 500L.}

\item{a_sigma}{A real number for the shape parameter in the Gamma prior of sigma2. The default value is 0.01.}

\item{b_sigma}{A real number for the rate parameter in the Gamma prior of sigma2. The default value is 0.01.}

\item{a_zeta}{A real number for the shape parameter in the Gamma prior of zeta. The default value is 0.01.}

\item{b_zeta}{A real number for the rate parameter in the Gamma prior of zeta. The default value is 0.01.}

\item{center}{A vector of real numbers specifying the centroid parameters in the modified exponential squared kernel. The default value is NULL and set to the center of the grid points: apply(x,2,mean).}

\item{scale}{A vector of positive numbers specifying the scale parameters in the modified exponential squared kernel. The default value is NULL and set to values such that grid points in a range of (-max_range,max_range) in each dimension.}

\item{max_range}{A positive real number indicating the maximum range of the grid points to specify the scale parameter. The default value is NULL and set to 6.}

\item{progress_bar}{A logical value to indicate whether a progress bar will be shown.}
}
\value{
A list of variables including the model fitting results
\describe{
 \item{f}{A vector of real numbers for the posterior mean of the fitted curve.}
 \item{x}{A matrix of real numbers for the grid points where rows are observations and columns are coordinates.}
 \item{work_x}{A matrix of real numbers for the standardized grid points for the model fitting. It has the same dimension as "x".}
 \item{sigma2}{A real number for the posterior mean of the variance parameter of random errors.}
 \item{tau2}{A real number for the posterior mean of the variance parameter for the Gaussian process prior.}
 \item{theta}{A vector of real numbers for the posterior mean of the basis coefficients for the Gaussian process.}
 \item{Xmat}{A matrix real numbers for the basis functions evaluated at the standardized grid points (work_x), where rows are observations and columns are the basis functions}
 \item{grid_size}{A real scalar for the grid size}
 \item{center}{A vector of real numbers for the centroid parameters in the modified exponential squared kernel.}
 \item{scale}{A vector of positive numbers for the scale parameters in the modified exponential squared kernel.}
 \item{max_range}{A positive real number indicating the maximum range of the grid points to specify the scale parameter.}
 \item{poly_degree}{An integer number to specify the highest degree of Hermite polynomials.}
 \item{a}{A positive real number to specify the concentration parameter in the standard modified exponential squared kernel.}
 \item{b}{A positive real number to specify the smoothness parameter in the standard modified exponential squared kernel.}
 \item{mcmc_sample}{A matrix of real numbers for saved MCMC samples.}
 \item{elapsed}{A real number indicating the computing time in second.}
}
}
\description{
Regular Bayesian fitting of Gaussian process regression on regular grid points with the modified exponential sqaured kernel.
}
\examples{
library(BayesGPfit)
library(lattice)
set.seed(1227)
dat = list()
dat$x = GP.generate.grids(d=2,num_grids = 100)
curve = GP.simulate.curve.fast(dat$x,a=0.01,b=0.5,poly_degree=20L)
dat$f = curve$f + rnorm(length(curve$f),sd=1)
fast_fit = GP.fast.Bayes.fit(dat$f,dat$x,a=0.01,b=0.5,poly_degree=20L,progress_bar = TRUE)
reg_fit = GP.Bayes.fit(dat$f,dat$x,a=0.01,b=0.5,poly_degree=20L,progress_bar = TRUE)
mse = c(reg = mean((reg_fit$f - curve$f)^2),
       fast = mean((fast_fit$f - curve$f)^2))
print(mse)
plot(GP.plot.curve(curve,main="True curve"),split=c(1,2,2,2),more=TRUE)
plot(GP.plot.curve(fast_fit,main="Posterior mean estimates (fast)"),split=c(2,2,2,2),more=TRUE)
plot(GP.plot.curve(reg_fit,main="Posterior mean estimates (Regular)"),split=c(2,1,2,2))
}
\author{
Jian Kang <jiankang@umich.edu>
}
