/* Copyright (C) 1997-1999  Adrian Trapletti
  
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   Linear algebra package for C */


#include <math.h>
#include "nrutil.h"
#include "nrlinal.h"


#include "nr.c"  /* brute force approach to hide the Numerical Recipes in the executable */


void vEQvPLv (double *x, double *y, double *z, long n)  
     /* x[i] = y[i]+z[i], i = 1,..,n, where x[1..n], y[1..n], and z[1..n] 
	x is allowed to be y or z */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]+z[i];
}

void vEQvMIv (double *x, double *y, double *z, long n)  
     /* x[i] = y[i]-z[i], i = 1,..,n, where x[1..n], y[1..n], and z[1..n] 
	x is allowed to be y or z */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]-z[i];
}

void vEQvTIv (double *x, double *y, double *z, long n)  
     /* x[i] = y[i]*z[i], i = 1,..,n, where x[1..n], y[1..n], and z[1..n] 
	x is allowed to be y or z */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]*z[i];
}
    
void vEQvDIv (double *x, double *y, double *z, long n)  
     /* x[i] = y[i]/z[i], i = 1,..,n, where x[1..n], y[1..n], and z[1..n] 
	x is allowed to be y or z */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]/z[i];
}
    
void vPLEQv (double *x, double *y, long n)  
     /* x[i] += y[i], i = 1,..,n, where x[1..n] and y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] += y[i];
}

void vMIEQv (double *x, double *y, long n)  
     /* x[i] -= y[i], i = 1,..,n, where x[1..n] and y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] -= y[i];
}

void vTIEQv (double *x, double *y, long n)  
     /* x[i] *= y[i], i = 1,..,n, where x[1..n] and y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] *= y[i];
}

void vDIEQv (double *x, double *y, long n)  
     /* x[i] /= y[i], i = 1,..,n, where x[1..n] and y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] /= y[i];
}

void vPLEQd (double *x, double y, long n)  
     /* x[i] += y, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] += y;
}

void vMIEQd (double *x, double y, long n)  
     /* x[i] -= y, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] -= y;
}

void vTIEQd (double *x, double y, long n)  
     /* x[i] *= y, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] *= y;
}

void vDIEQd (double *x, double y, long n)  
     /* x[i] /= y, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] /= y;
}

void vEQvPLd (double *x, double *y, double z, long n)  
     /* x[i] = y[i]+z, i = 1,..,n, where x[1..n], y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]+z;
}

void vEQvMId (double *x, double *y, double z, long n)  
     /* x[i] = y[i]-z, i = 1,..,n, where x[1..n], y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]-z;
}

void vEQvTId (double *x, double *y, double z, long n)  
     /* x[i] = y[i]*z, i = 1,..,n, where x[1..n], y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]*z;
}

void vEQvDId (double *x, double *y, double z, long n)  
     /* x[i] = y[i]/z, i = 1,..,n, where x[1..n], y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] = y[i]/z;
}

void ZEROv (double *x, long n)  
     /* x[i] = 0.0, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] = 0.0;
}

void ONEv (double *x, long n)  
     /* x[i] = 1.0, i = 1,..,n, where x[1..n] */
{
  long i;

  for (i=1; i<=n; i++) x[i] = 1.0;
}

void vEQMIv (double *x, double *y, long n)
     /* x[i] = -y[i], i = 1,..,n, where x[1..n], y[1..n] 
	x is allowed to be y */
{
  long i;

  for (i=1; i<=n; i++) x[i] = -y[i];
}

void vEQmTIv (double *x, double **y, double *z, long n, long m)  
     /* x = y*z, where x[1..n], y[1..n][1..m], and z[1..m] 
	x must be different from z */
{
  long i, j;
  double sum;

  for (i=1; i<=n; i++) 
  {
    sum = 0.0;
    for (j=1; j<=m; j++) sum += y[i][j]*z[j];
    x[i] = sum;
  }
}

void mEQmPLm (double **x, double **y, double **z, long n, long m)  
     /* x[i][j] = y[i][j]+z[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m], y[1..n][1..m], and z[1..n][1..m] 
	x is allowed to be y or z */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]+z[i][j];
}

void mEQmMIm (double **x, double **y, double **z, long n, long m)  
     /* x[i][j] = y[i][j]-z[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m], y[1..n][1..m], and z[1..n][1..m] 
	x is allowed to be y or z */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]-z[i][j];
}

void mEQmTIm (double **x, double **y, double **z, long n, long m)  
     /* x[i][j] = y[i][j]*z[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m], y[1..n][1..m], and z[1..n][1..m] 
	x is allowed to be y or z */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]*z[i][j];
}

void mEQmDIm (double **x, double **y, double **z, long n, long m)  
     /* x[i][j] = y[i][j]/z[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m], y[1..n][1..m], and z[1..n][1..m] 
	x is allowed to be y or z */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]/z[i][j];
}

void mEQmMTIm (double **x, double **y, double **z, long n, long m, long p)  
     /* x = y*z, where x[1..n][1..p], y[1..n][1..m], and z[1..m][1..p] 
	x must be different from y and z */
{
  long i, j, k;
  double sum;

  for (i=1; i<=n; i++) 
  {
    for (j=1; j<=p; j++) 
    {
      sum = 0.0;
      for (k=1; k<=m; k++) sum += y[i][k]*z[k][j];
      x[i][j] = sum;
    }
  }
}

void mPLEQm (double **x, double **y, long n, long m)  
     /* x[i][j] += y[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] += y[i][j];
}

void mMIEQm (double **x, double **y, long n, long m)  
     /* x[i][j] -= y[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] -= y[i][j];
}

void mTIEQm (double **x, double **y, long n, long m)  
     /* x[i][j] *= y[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] *= y[i][j];
}

void mDIEQm (double **x, double **y, long n, long m)  
     /* x[i][j] /= y[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] /= y[i][j];
}

void mPLEQd (double **x, double y, long n, long m)  
     /* x[i][j] += y, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] += y;
}

void mMIEQd (double **x, double y, long n, long m)  
     /* x[i][j] -= y, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] -= y;
}

void mTIEQd (double **x, double y, long n, long m)  
     /* x[i][j] *= y, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] *= y;
}

void mDIEQd (double **x, double y, long n, long m)  
     /* x[i][j] /= y, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] /= y;
}

void mEQmPLd (double **x, double **y, double z, long n, long m)  
     /* x[i][j] = y[i][j]+z, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]+z;
}

void mEQmMId (double **x, double **y, double z, long n, long m)  
     /* x[i][j] = y[i][j]-z, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]-z;
}

void mEQmTId (double **x, double **y, double z, long n, long m)  
     /* x[i][j] = y[i][j]*z, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]*z;
}

void mEQmDId (double **x, double **y, double z, long n, long m)  
     /* x[i][j] = y[i][j]/z, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = y[i][j]/z;
}

void mEQMIm (double **x, double **y, long n, long m)  
     /* x[i][j] = -y[i][j], i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] and y[1..n][1..m] 
	x is allowed to be y */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = -y[i][j];
}

double INNER (double *x, double *y, long n)
     /* Compute the inner product of x and y, i.e. x'*y, where x[1..n] and y[1..n] */
{
   long i;
   double sum;

   sum = 0.0;
   for (i=1; i<=n; i++) sum += x[i]*y[i];
   return sum;
} 

void OUTER (double **x, double *y, double *z, long n)
     /* Compute the outer product of y and z, i.e. x = y*z', 
	where x[1..n][1..n], y[1..n], and z[1..n] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=n; j++) 
      x[i][j] = y[i]*z[j];
}

double SUMv (double *x, long n)
     /* Compute the sum over the elements of x[1..n] */
{
   long i;
   double sum;

   sum = 0.0;
   for (i=1; i<=n; i++) sum += x[i];
   return sum;
} 

double MEANv (double *x, long n)
     /* Compute the mean over the elements of x[1..n] */
{
   long i;
   double sum;

   sum = 0.0;
   for (i=1; i<=n; i++) sum += x[i];
   sum /= (double) n;
   return sum;
} 

void RANDv (double *x, long n, int type, double mean, double var)
{
  long i;
  double range = var-mean;
  double std = sqrt(var);

  if (type == GAUSS)
    for (i=1; i<=n; i++) x[i] = std*gasdev()+mean;
  else if (type == UNIFORM)
    for (i=1; i<=n; i++) x[i] = range*ran2()+mean;
  else
    nrerror("not correct type in RANDv()");
}

void RANDm (double **x, long n, long m, int type, double mean, double var)
{
  long i, j;
  double range = var-mean;
  double std = sqrt(var);

  if (type == GAUSS)
    for (i=1; i<=n; i++)
      for (j=1; j<=m; j++)
	x[i][j] = std*gasdev()+mean;
  else if (type == UNIFORM)
    for (i=1; i<=n; i++)
      for (j=1; j<=m; j++)
	x[i][j] = range*ran2()+mean;
  else
    nrerror("not correct type in RANDm()");
}

void ZEROm (double **x, long n, long m)  
     /* x[i][j] = 0.0, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = 0.0;
}

void ONEm (double **x, long n, long m)  
     /* x[i][j] = 1.0, i = 1,..,n, j = 1,..,m, 
	where x[1..n][1..m] */
{
  long i, j;

  for (i=1; i<=n; i++) 
    for (j=1; j<=m; j++) 
      x[i][j] = 1.0;
}

void EYEm (double **x, long n)  
     /* x[i][i] = 1.0, i = 1,..,n, x[i][j] = 0.0, otherwise,
	where x[1..n][1..n] */
{
  long i, j;

  for (i=1; i<=n; i++) 
  {
    for (j=1; j<=n; j++) 
      x[i][j] = 0.0;
    x[i][i] = 1.0;
  }
}

