## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

library(admiraldev)

## ----warning=FALSE, message=FALSE---------------------------------------------
library(admiral)
library(admiralneuro)
library(pharmaversesdtm)
library(dplyr)
library(lubridate)
library(stringr)

## ----message=FALSE, warning=FALSE---------------------------------------------
nv <- pharmaversesdtm::nv_neuro %>%
  convert_blanks_to_na()

adsl <- admiralneuro::adsl_neuro %>%
  convert_blanks_to_na()

## ----message=FALSE, warning=FALSE---------------------------------------------
adsl_vars <- exprs(AGE, SEX, TRTSDT, TRTEDT, TRT01A, TRT01P)

adnv <- nv %>%
  derive_vars_merged(
    dataset_add = adsl,
    new_vars = adsl_vars,
    by_vars = exprs(STUDYID, USUBJID)
  ) %>%
  filter(NVTESTCD == "UPSIT") %>%
  derive_vars_dt(
    new_vars_prefix = "A",
    dtc = NVDTC
  ) %>%
  derive_vars_dy(reference_date = TRTSDT, source_vars = exprs(ADT))

## ----echo=TRUE, message=FALSE-------------------------------------------------
param_lookup <- tibble::tribble(
  ~NVTESTCD, ~NVCAT, ~PARAMCD, ~PARAM, ~PARAMN,
  "UPSIT", "OLFACTORY FUNCTION", "UPSITTS", "UPSIT Combined Score from 40 Odorant", 1
)

## ----eval=TRUE, include=TRUE, message=FALSE-----------------------------------
adnv <- adnv %>%
  derive_vars_merged_lookup(
    dataset_add = param_lookup,
    new_vars = exprs(PARAMCD, PARAM, PARAMN),
    by_vars = exprs(NVTESTCD, NVCAT)
  )

## ----eval=TRUE, echo=FALSE----------------------------------------------------
adnv_param <- distinct(adnv, USUBJID, NVTESTCD, NVCAT, PARAMCD, PARAM, PARAMN)
dataset_vignette(adnv_param, display_vars = exprs(USUBJID, NVTESTCD, NVCAT, PARAMCD, PARAM, PARAMN))

## ----message=FALSE, warning=FALSE---------------------------------------------
adnv <- adnv %>%
  mutate(
    AVAL = NVSTRESN,
    AVALC = if_else(
      is.na(NVSTRESN) | as.character(NVSTRESN) != NVSTRESC,
      NVSTRESC,
      NA
    )
  )

## ----message=FALSE, warning=FALSE---------------------------------------------
if (all(is.na(adnv$AVALC))) {
  adnv <- adnv %>% select(-AVALC)
}

## ----message=FALSE, warning=FALSE---------------------------------------------
keep_vars <- c(
  get_admiral_option("subject_keys"),
  adsl_vars,
  exprs(ADT, ADY, VISIT)
)

# Prepare derived rows
upsit_percentile <- adnv %>%
  filter(PARAMCD == "UPSITTS") %>%
  mutate(
    AVAL = compute_upsit_percentile(
      sex = SEX,
      age = AGE,
      upsit_score = AVAL
    ),
    PARAMCD = "UPSITPC",
    PARAM = "Percentile derived from UPSIT total score",
    PARAMN = 2,
    AVALU = NA
  )

# Bind with original dataset
adnv <- bind_rows(adnv, upsit_percentile)

# Drop AVALU because it contains no non-missing values
if (all(is.na(adnv$AVALU))) {
  adnv <- adnv %>% select(-AVALU)
}

## ----eval=TRUE, echo=FALSE----------------------------------------------------
adnv_param <- distinct(adnv, USUBJID, PARAMCD, PARAM, AVAL, ADT, ADY, VISIT)
dataset_vignette(adnv_param %>% arrange(USUBJID, ADT), display_vars = exprs(USUBJID, PARAMCD, PARAM, AVAL, ADT, ADY, VISIT))

## ----message=FALSE, warning=FALSE---------------------------------------------
adnv <- adnv %>%
  restrict_derivation(
    derivation = derive_vars_crit_flag,
    args = params(
      crit_nr = 1,
      condition = if_else(PARAMCD == "UPSITPC", AVAL <= 10, NA),
      description = "UPSITPC <= 10",
      values_yn = TRUE # To get "Y", "N", and NA for the flag
    ),
    filter = PARAMCD == "UPSITPC"
  )

## ----eval=TRUE, echo=FALSE----------------------------------------------------
adnv_flags <- distinct(adnv, USUBJID, PARAMCD, PARAM, AVAL, ADT, ADY, VISIT, CRIT1, CRIT1FL)
dataset_vignette(adnv_flags %>% arrange(USUBJID, ADT), display_vars = exprs(USUBJID, PARAMCD, PARAM, AVAL, ADT, ADY, VISIT, CRIT1, CRIT1FL))

