## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 4
)

## ----setup--------------------------------------------------------------------
library(fasster)

## -----------------------------------------------------------------------------
library(dplyr)
library(lubridate)

# Use Australian gas production data
gas_data <- tsibbledata::aus_production |>
  select(Quarter, Gas) |>
  filter(!is.na(Gas))

# Split into training and test sets
train <- gas_data |> filter(year(Quarter) < 2008)
test <- gas_data |> filter(year(Quarter) >= 2008)

## -----------------------------------------------------------------------------
# Fit model with pre/post 1970 trends
fit <- train |>
  model(
    fasster = FASSTER(log(Gas) ~ (year(Quarter) < 1970) %S% trend(2) + fourier("year"))
  )

fit

## -----------------------------------------------------------------------------
components(fit)

# Plot decomposition
components(fit) |> 
  autoplot()

## -----------------------------------------------------------------------------
# Forecast test period
fc <- forecast(fit, h = "5 years")

# Plot forecasts
autoplot(fc, train)

## -----------------------------------------------------------------------------
# Model works even with missing values
data_with_na <- gas_data |>
  # Insert some missing values randomly
  mutate(Gas = if_else(row_number() %in% sample(n(), 10), NA_real_, Gas))

fit_na <- data_with_na |>
  model(FASSTER(Gas ~ (year(Quarter) < 1970) %S% trend(2) + fourier("year", K = 2)))

# Interpolate missing values
fit_na |> 
  interpolate(data_with_na)

