################################################################################
# Performs immediate Tang-Tang-Chan test for equivalence in paired binary data
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9 
# Date: Feb 06, 2026

tost.rrpi <- function(
    a=NA, b=NA, c=NA, n=NA,
    delta0=1, deltaupper=NA,
    exact.chisq=FALSE,
    conf.level=0.95,
    treatment1="", treatment2="", outcome="", nooutcome="",
    relevance=TRUE) {

  # Validate that a through d are positive integers
  if (is.na(a) | a%%1 != 0 | a < 0) {
    stop("a must be a non-negative integer")
    }
  if (is.na(b) | b%%1 != 0 | b < 0) {
    stop("b must be a non-negative integer")
    }
  if (is.na(c) | c%%1 != 0 | c < 0) {
    stop("c must be a non-negative integer")
    }
  if (is.na(n) | n%%1 != 0 | n < 1 | n < (a + b + c)) {
    stop("n must be a positive integer \U2265 a+b+c")
    }
  d <- n - a - b - c
  if (is.na(delta0) | !is.numeric(delta0) | delta0 < 0 | delta0 > 1) {
    stop("delta0 must be a positive real value that is less than or equal to 1")
    }

# Validate deltaupper
  if (!is.na(deltaupper) & (!is.numeric(deltaupper) | deltaupper < 1)) {
    stop("deltaupper must be a positive real value that is greater than or equal to 1")
    }
   else {
    if (!is.na(deltaupper)) {
      upper <- 1/deltaupper
      }
	   }
  if (is.na(deltaupper)) {
    upper <- delta0
    }
  lower <- delta0

  # Validate conf.level and create alpha
  alpha <- 1 - conf.level
  if (alpha <= 0 | alpha >= 1) {
    stop("conf.level must be >0 and <1")
    }
  alpha.display <- sprintf("%-8.4g", alpha)
   
  # Set up treatment and outcome status names, as user requests
  if (treatment1=="") {
    treatname1 <- "Treatment 1"
   	}
   else {
    treatname1 <- treatment1
	   }
  treatname1     <- substr(treatname1,1,9)
  treatname1long <- substr(treatment1,1,16)

  if (treatment2=="") {
    treatname2 <- "Treatment 2"
   	}
   else {
    treatname2 <- treatment2
   	}
  treatname2     <- substr(treatname2,1,9)
  treatname2long <- substr(treatment2,1,16)

  if (outcome=="") {
    positive    <- "Positive"
    rowpositive <- "Positive"
    colpositive <- "Positive"
   	}
   else {
    positive    <- substr(outcome,1,9)
    rowpositive <- substr(outcome,1,16)
    colpositive <- substr(outcome,1,9)
   	}
  if (nooutcome=="") {
    negative    <- "Negative"
    rownegative <- "Negative"
    colnegative <- "Negative"
	   }
   else {
    negative    <- substr(nooutcome,1,9)
    rownegative <- substr(nooutcome,1,16)
    colnegative <- substr(nooutcome,1,9)
   	}
 

###############################################################################
# The business starts here                                                    #
###############################################################################
  PositivistConclusion <- "Reject"
  NegativistConclusion <- "Reject"

##########
# Do McNemar's test for proportion difference (because if OR != 1, RR != 1)
  chisq.pos <- ((b - c)^2)/(b + c)  
  chisq.pos.display <- pad.left(sprintf("%8.4f",chisq.pos),11)
  p.pos <- pchisq(chisq.pos, df=1, lower.tail=FALSE)
  chisq_p_val_notice <- ""
  if (exact.chisq) {
    p.pos <- 2*pbinom(q=min(b,c), size=b + c, prob=0.5)
    chisq_p_val_notice <- "Using exact p-value for McNemar\U2019s \u03C7\u2072\n"
    }
  p.pos.display <- format.extreme.p.vals(p.pos)
  if (p.pos > alpha/2) {
    PositivistConclusion <- "Fail to reject"
    }

  # Create top bar, mid bar, and bottom bar
  top.bar    <- paste0(
       paste0(rep("\U2500",17), collapse="", sep=""), 
       "\U252C",
       paste0(rep("\U2500",24), collapse="", sep=""),
       "\U252C",
       paste0(rep("\U2500",12), collapse="", sep=""),
       collapse="")
  table.title <- paste0(
       pad.spaces(17),
       "\U2502  ",
       pad.center(treatname1long,21),
       " \U2502\n",
       pad.right(treatname2long, 16),
       " \U2502 ",
       pad.left(positive,9),
       pad.spaces(3),
       pad.left(negative,9),
       pad.spaces(2),
       "\U2502",
       pad.spaces(6),
       "Total", collapse="") 
  mid.bar    <- paste0(
       paste0(rep("\U2500",17), collapse="", sep=""), 
       "\U253C",
       paste0(rep("\U2500",24), collapse="", sep=""),
       "\U253C",
       paste0(rep("\U2500",12), collapse="", sep=""),
       collapse="")
  bottom.bar <- paste0(
       paste0(rep("\U2500",17), collapse="", sep=""), 
       "\U2534",
       paste0(rep("\U2500",24), collapse="", sep=""),
       "\U2534",
       paste0(rep("\U2500",12), collapse="", sep=""),
       collapse="")

  if (relevance) {
    rlang::inform(message="\nRelevance test for relative risk and unity in paired designs\n")
    level <- 100*(1 - alpha)
    rlang::inform(message=paste0("McNemar\U2019s test for difference between relative risk and unity in paired designs"))
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(
          pad.left(rowpositive,16),
          " \U2502 ", 
          pad.left(a,9,strip0=FALSE),
          pad.spaces(3),
          pad.left(b,9,strip0=FALSE),
          "  \U2502 ",
          pad.left(a+b,9,strip0=FALSE),
          collapse=""))
    rlang::inform(message=paste0(
          pad.left(rownegative,16),
          " \U2502 ", 
          pad.left(c,9,strip0=FALSE),
          pad.spaces(3),
          pad.left(d,9,strip0=FALSE),
          "  \U2502 ",
          pad.left(c+d,9,strip0=FALSE),
          collapse=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(
          pad.left("Total",16),
          " \U2502 ", 
          pad.left(a+c,9,strip0=FALSE),
          pad.spaces(3),
          pad.left(b+d,9,strip0=FALSE),
          "  \U2502 ",
          pad.left(a + b + c + d,9,strip0=FALSE),
          collapse=""))
    rlang::inform(message=bottom.bar)
    rlang::inform(message=paste0("McNemar\U2019s \u03C7\u2072(1) = ", chisq.pos.display, pad.spaces(8),"Pr(X > \U03C7\U2072) ", p.pos.display,sep=""))
    rlang::inform(message=paste0(chisq_p_val_notice,"Reject Ho if p \u2264 ", 1 - conf.level, "/2 (two-sided test)\n", sep=""))
    rlang::inform(message=paste0("Ho: Pr(",positive,"|",treatname1,") = Pr(",positive,"|",treatname2,")", sep=""))
    rlang::inform(message=paste0("Ha: Pr(",positive,"|",treatname1,") \U2260 Pr(",positive,"|",treatname2,")\n", sep=""))
    }
  if (a==0 & b==0 & c==0) {
    RR <- NA
	   sdRR <- NA
    z1 <- NA
	   z2 <- NA
	   rlang::inform(message="\nNote: neither treatment has any observed responses; relative risk undefined")
	   }
   else if (a>0 & b==0 & c==0) {
    RR <- 1
   	riskprefix <- ""
   	sdRR <- sqrt((a + b) * (b + c) / ((a + c)^3))
    z1 <- sqrt(a*(1-lower)/lower)
    z2 <- sqrt(a*(1-upper)/upper)
    rlang::inform(message="\nNote: treatments have complete concordance")
    }
   else {
    RR <- (a + b) / (a + c)
   	riskprefix <- "0"
	   sdRR <- sqrt((a + b) * (b + c)/((a + c)^3))
    z1 <- TangTangChanZ(a, b, c, n, lower)
    z2 <- TangTangChanZ(a, c, b, n, upper)
    }
  p1 <- pnorm(z1, lower.tail=FALSE)
  p2 <- pnorm(z2, lower.tail=FALSE)
  p1.display <- format.extreme.p.vals(p1)
  p2.display <- format.extreme.p.vals(p2)
  if (p1 > alpha | p2 > alpha) {
    NegativistConclusion <- "Fail to reject"
    }
  rlang::inform(message="\nTang-Tang-Chan test for equivalence of relative risk and unity in paired designs")
  rlang::inform(message=top.bar)
  rlang::inform(message=table.title)
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(
        pad.left(rowpositive,16),
        " \U2502 ", 
        pad.left(a,9,strip0=FALSE),
        pad.spaces(3),
        pad.left(b,9,strip0=FALSE),
        "  \U2502 ",
        pad.left(a+b,9,strip0=FALSE),
        collapse=""))
  rlang::inform(message=paste0(
        pad.left(rownegative,16),
        " \U2502 ", 
        pad.left(c,9,strip0=FALSE),
        pad.spaces(3),
        pad.left(d,9,strip0=FALSE),
        "  \U2502 ",
        pad.left(c+d,9,strip0=FALSE),
        collapse=""))
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(
        pad.left("Total",16),
        " \U2502 ", 
        pad.left(a+c,9,strip0=FALSE),
        pad.spaces(3),
        pad.left(b+d,9,strip0=FALSE),
        "  \U2502 ",
        pad.left(a+b+c+d,9,strip0=FALSE),
        collapse=""))
  rlang::inform(message=bottom.bar)
  rlang::inform(message=paste0("Relative risk of ",positive, " in ",treatname2, " vs. ", treatname1, ":", sep=""))
  rlang::inform(message=paste0(pad.spaces(8), "RR = ", a+b, " / ", a+c, " = ", riskprefix, trimws(sprintf("%-8.4g", RR)), sep=""))
  rlang::inform(message=paste0(pad.spaces(3), "s.e. RR = ", trimws(sprintf("%-8.4g", sdRR)), sep=""))
  rlang::inform(message=paste0(pad.spaces(8), "\u03B4l = ", trimws(sprintf("%-8.4f", lower)), sep=""))
  rlang::inform(message=paste0(pad.spaces(8), "\u03B4u = ", trimws(sprintf("%-8.4f", 1/upper)), sep=""))

  rlang::inform(message=paste0("\nHo: true RR \u2264 \u03B4l, or true RR \u2265 \u03B4u:\n")) 
  rlang::inform(message=paste0(pad.spaces(8), "z1 = ", trimws(sprintf("%-8.4g", z1)), pad.spaces(19), "z2 = ", trimws(sprintf("%-8.4g", z2)), sep=""))
  rlang::inform(message=paste0(pad.spaces(3), "Ho1: true RR \u2264 \u03B4l", pad.spaces(12), "Ho2: true RR \u2265 \u03B4u", sep=""))
  rlang::inform(message=paste0(pad.spaces(3), "Ha1: true RR > \u03B4l", pad.spaces(12), "Ha2: true RR < \u03B4u", sep=""))
  rlang::inform(message=paste0(pad.spaces(3), "Pr(Z > z1) ", p1.display, pad.spaces(10), "Pr(Z > z2) ", p2.display, sep=""))


# Output combined tests results if relevance test is requested
 if (relevance) {
   if (upper == lower) {
     rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display, ", \u03B4l = ", trimws(sprintf("%5.2f", lower)), ", and \u03B4u = ", trimws(sprintf("%5.2f", 1/lower)), ":", sep=""))
     }
   else {
     rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display, ", \u03B4l = ", trimws(sprintf("%5.2f", lower)), ", and \u03B4u = ", trimws(sprintf("%5.2f", 1/upper)), ":", sep=""))
     }
   rlang::inform(message=paste0("  Ho test for difference:  ",PositivistConclusion)) 
   rlang::inform(message=paste0("  Ho test for equivalence: ",NegativistConclusion))
   if (PositivistConclusion == "Reject" & NegativistConclusion == "Reject") {
     rel.conclusion <- "Trivial difference (overpowered test)"
     }
   if (PositivistConclusion == "Reject" & NegativistConclusion == "Fail to reject") {
     rel.conclusion <- "Relevant difference"
     }
   if (PositivistConclusion == "Fail to reject" & NegativistConclusion == "Reject") {
     rel.conclusion <- "Equivalence"
     }
   if (PositivistConclusion == "Fail to reject" & NegativistConclusion == "Fail to reject") {
     rel.conclusion <- "Indeterminate (underpowered test)"
     }
   rlang::inform(message=paste0("\nConclusion from combined tests: ",rel.conclusion,sep=""))
   }


###############################################################################
# Program end. Close up shop and return things.                               #
###############################################################################
  out <- list() 
  # Prepare return stuff
  if (!relevance) {
    out$statistics <- c(z1,z2)
    names(out$statistics) <- c("z1","z2")
    out$p.values <- c(p1,p2)
    names(out$p.values) <- c("p1","p2")
    }
   else {
    out$statistics <- c(z1,z2,chisq.pos)
    names(out$statistics) <- c("z1","z2",paste0("\U03C7\U00B2",sep=""))
    out$p.values <- c(p1,p2,p.pos)
    if (exact.chisq) {
      names(out$p.values) <- c("p1","p2","p exact")
      }
    else {
      names(out$p.values) <- c("p1","p2","p")
      }
    }
  out$estimate <- RR
  names(out$estimate) <- "RR"
  out$error <- sdRR
  names(out$error) <- "std RR"
  if (upper==lower) {
    out$threshold <- lower
    names(out$threshold) <- "\U03B4\U2080"
    }
   else {
    out$threshold <- c(lower, upper)
    names(out$threshold) <- c("\U03B4l", "\U03B4u")
    }
  if(relevance) {
    out$conclusion <- rel.conclusion
    names(out$conclusion) <- "relevance conclusion"
    }
  invisible(out)
  }

# q1 returns the primary root of the "q1" term of f(x) at the top of page 1221
# in Tang, Tang, and Chan (2003)
q1 <- function(a, b, c, n, threshold) {
  A <- n * ( 1 + threshold)
  B <- (a + c) * (threshold^2) - (a + b + 2 * c)
  C <- c * (1 - threshold) * (a + b + c) / n
  q1 <- (sqrt((B^2) - (4 * A * C)) - B) / (2 * A)
  return(q1)
  }
  
TangTangChanZ <- function(a, b, c, n, threshold) {
  numerator <- a + b - (a + c) * threshold
  Q1 <- q1(a, b, c, n, threshold)
  denominator <- sqrt( n * ((1 + threshold)*Q1 + ((a + b + c) * (threshold - 1) / n)))
  TTC <- numerator/denominator
  return(TTC)
  }
