
#' Coefficients from an snreg Model
#'
#' @title Extract Model Coefficients
#'
#' @description
#' \code{coef.snreg} is the S3 method for extracting the estimated regression
#' coefficients from an object of class \code{"snreg"}.
#'
#' @param object
#' an object of class \code{"snreg"}, typically returned by \code{\link{snreg}}.
#'
#' @param ...
#' additional arguments (currently unused).
#'
#' @return
#' A numeric vector containing the model coefficients.
#'
#' @details
#' This method simply returns the \code{coef} component stored inside the fitted
#' \code{"snreg"} object. If the object does not contain coefficient estimates
#' (e.g., if estimation was not completed in a scaffold), an informative error
#' is raised.
#'
#' @seealso
#' \code{\link{snsf}}, \code{\link{snreg}}, \code{\link{lm.mle}}, \code{\link{vcov.snreg}}, \code{\link{residuals.snreg}}
#' 
#' @examples
#' library(snreg)
#' 
#' data("banks07")
#' head(banks07)
#' 
#' # Translog cost function specification
#' 
#' spe.tl <- log(TC) ~ (log(Y1) + log(Y2) + log(W1) + log(W2))^2 +
#'   I(0.5 * log(Y1)^2) + I(0.5 * log(Y2)^2) +
#'   I(0.5 * log(W1)^2) + I(0.5 * log(W2)^2)
#' 
#' # Specification 1: homoskedastic noise and skewness
#' 
#' formSV <- NULL   # variance equation; constant variance
#' formSK <- NULL   # skewness equation; constant skewness
#' 
#' m1 <- snreg(
#'   formula  = spe.tl,
#'   data     = banks07,
#'   ln.var.v = formSV,
#'   skew.v   = formSK
#' )
#' 
#' coef(m1)
#'
#' @export
coef.snreg <- function(object, ...) {
  if (is.null(object)) {
    stop("Argument 'object' is NULL; expected a fitted 'snreg' objectect.", call. = FALSE)
  }
  if (is.null(object$coef)) {
    stop("No coefficients available in 'object$coef'.", call. = FALSE)
  }
  object$coef
}

# coef.snreg <- function( obj, ... ) {
#   if(is.null(obj$coef) | is.null(obj)){
#     stop( paste("No results are available in ",obj,"") )
#   }
#   return( obj$coef )
# }