% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hsr_readmission_risks.R
\name{hsr_readmission_risks}
\alias{hsr_readmission_risks}
\title{Compute discharge-level readmission risks in a Hospital-Specific Report (HSR)}
\usage{
hsr_readmission_risks(file, cohort)
}
\arguments{
\item{file}{File path to a report}

\item{cohort}{Cohort to compute readmission risks for. One of \code{c("AMI", "COPD", "HF", "PN", "CABG", "HK")}}
}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} containing the following columns:
\itemize{
\item \verb{ID Number}: The unique discharge identifier (see \code{\link[=hsr_discharges]{hsr_discharges()}})
\item \code{Predicted}: The predicted readmission risk for the discharge
\item \code{Expected}: The expected readmission risk for the discharge
}
}
\description{
Computes the \emph{predicted} and \emph{expected} readmission risks for each eligible discharge in the specified cohort.

\emph{\strong{Note}: CMS changed the format of Hospital-Specific Reports (HSRs) for FY2026 (see \href{https://qualitynet.cms.gov/inpatient/hrrp/reports#tab2}{here}). The current HSR functions support formats through FY2025.}
}
\details{
The \href{https://qualitynet.cms.gov/inpatient/measures/readmission}{readmission measure} is what \href{https://www.cms.gov/}{CMS} uses to grade performance
in the \href{https://www.cms.gov/medicare/payment/prospective-payment-systems/acute-inpatient-pps/hospital-readmissions-reduction-program-hrrp}{Hospital Readmissions Reduction Program (HRRP)}.

Individual discharges are assigned an adjusted readmission risk (based on clinical history), which then get aggregated into a
hospital-level score and compared to peer groups for penalty determination. Specifically, a random-intercept logistic regression model
is built for each cohort (\href{https://qualitynet.cms.gov/inpatient/measures/readmission/methodology}{see methodology}) which serves as the
basis for two (2) readmission risks assigned to each discharge:
\itemize{
\item \emph{Predicted}: Adjusted for patient-specific clinical factors plus the \emph{hospital-specific} effect (random intercept term)
\item \emph{Expected}: Adjusted for patient-specific clinical factors plus the \emph{hospital-average} effect
}

These quantities are then aggregated across all discharges and their ratio is taken to form the \emph{Excess Readmission Ratio (ERR)}, which
is then used as the cohort-specific comparison metric. Thus, it is a comparative measure of how likely patients are to be readmitted at
\emph{your} hospital versus the \emph{average} hospital, given your hospital's clinical characteristics.
}
\examples{
# Access a report
my_report <- hsr_mock_reports("FY2025_HRRP_MockHSR.xlsx")

# Compute readmission risks for HF discharges
hf_risks <- hsr_readmission_risks(my_report, "HF")
hf_risks

# Compute the ERR from scratch
hf_risks |>
 dplyr::summarize(
   Discharges = dplyr::n(),
   Predicted = mean(Predicted),
   Expected = mean(Expected),
   ERR = Predicted / Expected
 )


# Check that this matches the report table
hsr_cohort_summary(my_report) |>
 dplyr::select(
  dplyr::matches(
     paste0(
       "^Measure|",
       "^Number of Eligible Discharges|",
       "^Predicted|",
       "^Expected|",
       "^Excess"
     )
   )
 )
}
