% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/accessors-doc.R
\name{accessors}
\alias{accessors}
\title{Accessor functions for quallmer objects}
\description{
Functions to safely access and modify metadata from quallmer objects
(\code{qlm_coded}, \code{qlm_comparison}, \code{qlm_validation}, \code{qlm_codebook}). These
functions provide a stable API for accessing object metadata without
directly manipulating internal attributes.
}
\section{Metadata types}{


quallmer objects store metadata in three categories:

\strong{User metadata} (\code{type = "user"}):
\itemize{
\item \code{name}: Run identifier (settable)
\item \code{notes}: Descriptive notes (settable)
\item Plus any custom fields added via \code{as_qlm_coded(..., metadata = list(...))}
}

\strong{Object metadata} (\code{type = "object"}):
\itemize{
\item \code{call}: Function call that created the object
\item \code{parent}: Parent run name (for replications)
\item \code{batch}: Whether batch processing was used
\item \code{chat_args}: Arguments passed to the LLM chat
\item \code{execution_args}: Arguments for parallel/batch execution
\item \code{n_units}: Number of coded units
\item \code{input_type}: Type of input ("text", "image", or "human")
\item \code{source}: Coding source ("human" or "llm")
\item \code{is_gold}: Whether this is a gold standard
}

\strong{System metadata} (\code{type = "system"}):
\itemize{
\item \code{timestamp}: When the object was created
\item \code{ellmer_version}: Version of ellmer package
\item \code{quallmer_version}: Version of quallmer package
\item \code{R_version}: Version of R
}
}

\section{Functions}{

\itemize{
\item \code{\link[=qlm_meta]{qlm_meta()}}: Get metadata fields
\item \code{\link[=qlm_meta<-]{qlm_meta<-()}}: Set user metadata fields (only \code{name} and \code{notes})
\item \code{\link[=codebook]{codebook()}}: Extract codebook from coded objects
\item \code{\link[=inputs]{inputs()}}: Extract original input data
}
}

\examples{
\donttest{
# Create a coded object
texts <- c("I love this!", "Terrible.", "It's okay.")
coded <- qlm_code(
  texts,
  data_codebook_sentiment,
  model = "openai/gpt-4o-mini",
  name = "run1",
  notes = "Initial coding run"
)

# Access metadata
qlm_meta(coded, "name")              # Get run name
qlm_meta(coded, type = "user")       # Get all user metadata
qlm_meta(coded, type = "system")     # Get system metadata

# Modify user metadata
qlm_meta(coded, "name") <- "updated_run1"
qlm_meta(coded, "notes") <- "Revised notes"

# Extract components
codebook(coded)                      # Get the codebook
inputs(coded)                        # Get original texts

# Custom metadata from human coding
human_data <- data.frame(
  .id = 1:5,
  sentiment = c("pos", "neg", "pos", "neg", "pos")
)
human_coded <- as_qlm_coded(
  human_data,
  name = "coder_A",
  metadata = list(
    coder_name = "Dr. Smith",
    experience = "5 years"
  )
)

# Access custom metadata
qlm_meta(human_coded, "coder_name")  # "Dr. Smith"
qlm_meta(human_coded, type = "user") # All user fields
}

}
\seealso{
\itemize{
\item \code{\link[=qlm_code]{qlm_code()}} for creating coded objects
\item \code{\link[=as_qlm_coded]{as_qlm_coded()}} for converting human-coded data
\item \code{\link[=qlm_trail]{qlm_trail()}} for viewing coding history
}
}
