#' @title Non-Compartmental Analysis (NCA) of Plasma Concentration-Time Data (Nonlinear Form)
#' @name non_compartmental_nl
#' @description
#' Performs non-compartmental analysis (NCA) on plasma concentration-time data,
#' fitting the terminal elimination phase using nonlinear regression of an exponential
#' decay. Computes area under the curve (AUC) using trapezoidal rule, estimates
#' terminal elimination rate constant (kel), half-life (t1/2), clearance (CL),
#' and volume of distribution (Vd).
#'
#' @param data A data.frame containing plasma concentration-time data.
#' @param time_col Character string indicating the column name for time.
#' @param conc_col Character string indicating the column name for concentration.
#' @param dose Numeric value for the administered dose.
#' @param group_col Optional character string specifying a grouping variable for multiple groups.
#' @param terminal_points Number of last points to use for terminal phase estimation (default = 3).
#' @param plot Logical; if TRUE, plots concentration-time profile and terminal phase fit.
#' @param annotate Logical; if TRUE, annotates plot with PK parameters (only for <= 2 groups).
#'
#' @import scales
#' @import stats
#' @import ggplot2
#' @importFrom scales hue_pal
#' @importFrom stats na.omit lm optim
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text alpha scale_y_log10
#' labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with kel, t1/2, AUC, CL, Vd, and R^2 for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single-subject nonlinear non-compartmental analysis
#' df <- data.frame(
#'   time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(18.4, 16.9, 14.3, 10.9, 7.1, 4.7, 3.2, 2.0)
#' )
#' non_compartmental_nl(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   terminal_points = 3,
#'   plot = TRUE,
#'   annotate = TRUE
#' )
#'
#' # Example II: Two-group nonlinear NCA (e.g., formulation comparison)
#' df_groups <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     17.8, 16.3, 13.9, 10.5, 6.6, 4.3, 3.0,  # Group A
#'     20.1, 18.7, 16.2, 12.7, 8.4, 5.9, 4.1   # Group B
#'   ),
#'   formulation = rep(c("Reference", "Test"), each = 7)
#' )
#' non_compartmental_nl(
#'   data = df_groups,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "formulation",
#'   terminal_points = 3,
#'   plot = TRUE,
#'   annotate = TRUE
#' )
#'
#' # Example III: Six-subject nonlinear NCA
#' df_subjects <- data.frame(
#'   time = rep(c(0.5, 1, 2, 4, 8, 12, 24), 6),
#'   concentration = c(
#'     15.6, 14.1, 11.7, 7.9, 4.3, 2.8, 1.4,  # S1
#'     14.9, 13.5, 11.1, 7.4, 4.0, 2.6, 1.3,  # S2
#'     16.3, 14.9, 12.4, 8.4, 4.7, 3.1, 1.6,  # S3
#'     15.1, 13.7, 11.3, 7.6, 4.2, 2.7, 1.3,  # S4
#'     14.6, 13.2, 10.8, 7.2, 3.9, 2.5, 1.2,  # S5
#'     16.0, 14.6, 12.0, 8.1, 4.5, 3.0, 1.5   # S6
#'   ),
#'   subject = rep(paste0("S", 1:6), each = 7)
#' )
#' non_compartmental_nl(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 150,
#'   group_col = "subject",
#'   terminal_points = 4,
#'   plot = TRUE,
#'   annotate = FALSE
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics, 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded. Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "kel", "t_half", "C_fit", "AUC", "CL",
                         "Vd", "R2", "label", "x_pos", "y_pos", "hjust", "vjust",
                         "terminal_data", "t_last", "fit_vals", "head", "tail"))

non_compartmental_nl <- function(data,
                                 time_col = "time",
                                 conc_col = "concentration",
                                 dose,
                                 group_col = NULL,
                                 terminal_points = 3,
                                 plot = TRUE,
                                 annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")
  if (!requireNamespace("scales", quietly = TRUE))
    stop("Package 'scales' is required.")

  # Prepare data
  cols <- c(time_col, conc_col)
  if (!is.null(group_col)) cols <- c(cols, group_col)

  df <- data[, cols, drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  df <- df[df$time >= 0 & df$conc > 0, ]

  # Group handling (FORCED FACTOR)
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("Experimental")
  }

  # AUC (trapezoidal rule)
  calc_auc <- function(time, conc) {
    sum(diff(time) * (head(conc, -1) + tail(conc, -1)) / 2)
  }

  # Nonlinear terminal-phase fit
  fit_terminal_nl <- function(d) {
    if (nrow(d) < terminal_points)
      stop("Not enough points to estimate terminal phase.")

    terminal_data <- tail(d, terminal_points)
    t_last <- max(terminal_data$time)
    C_last <- terminal_data$conc[nrow(terminal_data)]

    # Initial kel from log-linear fit
    lin_fit <- stats::lm(log(conc) ~ time, data = terminal_data)
    init_kel <- max(1e-6, -coef(lin_fit)[2])

    # Objective function
    obj_fun <- function(par) {
      C_pred <- par[1] * exp(-par[2] * (terminal_data$time - t_last))
      sum((terminal_data$conc - C_pred)^2)
    }

    opt <- stats::optim(
      par = c(C_last, init_kel),
      fn = obj_fun,
      method = "L-BFGS-B",
      lower = c(0, 0)
    )

    C_fit <- opt$par[1]
    kel <- opt$par[2]
    t_half <- log(2) / kel

    auc <- calc_auc(d$time, d$conc)
    CL <- dose / auc
    Vd <- CL / kel

    fitted_vals <- C_fit * exp(-kel * (terminal_data$time - t_last))
    ss_res <- sum((terminal_data$conc - fitted_vals)^2)
    ss_tot <- sum((terminal_data$conc - mean(terminal_data$conc))^2)
    r2 <- 1 - ss_res / ss_tot

    list(
      kel = kel,
      t_half = t_half,
      C_fit = C_fit,
      auc = auc,
      CL = CL,
      Vd = Vd,
      r2 = r2,
      terminal_data = terminal_data,
      t_last = t_last
    )
  }

  # Per-group results
  results <- lapply(split(df, df$group), function(d) {
    term <- fit_terminal_nl(d)
    data.frame(
      group = unique(d$group),
      kel = term$kel,
      t_half = term$t_half,
      AUC = term$auc,
      CL = term$CL,
      Vd = term$Vd,
      R2 = term$r2
    )
  })

  fit_results <- do.call(rbind, results)

  # Plotting
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::scale_y_log10() +
      ggplot2::labs(
        title = "Non-Compartmental Analysis (Nonlinear Form)",
        subtitle = paste0("Terminal phase last ", terminal_points, " points"),
        x = "Time (hours)",
        y = "Plasma Concentration (log scale)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid = ggplot2::element_blank()
      )

    grp_levels <- levels(df$group)
    cols <- scales::hue_pal()(length(grp_levels))

    # Nonlinear terminal fits (dashed)
    for (i in seq_along(grp_levels)) {
      d_grp <- df[df$group == grp_levels[i], ]
      term <- fit_terminal_nl(d_grp)

      t_seq <- seq(
        min(term$terminal_data$time),
        max(term$terminal_data$time),
        length.out = 100
      )

      fit_vals <- term$C_fit * exp(-term$kel * (t_seq - term$t_last))

      df_fit <- data.frame(time = t_seq, conc = fit_vals)

      p <- p + ggplot2::geom_line(
        data = df_fit,
        ggplot2::aes(x = time, y = conc),
        color = ggplot2::alpha(cols[i], 0.7),
        linetype = "dashed",
        linewidth = 1
      )
    }

    # Annotation (<= 2 groups)
    if (annotate && nlevels(df$group) <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "AUC = ", round(ann$AUC, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "R2 = ", round(ann$R2, 3)
      )

      x_min <- min(df$time); x_max <- max(df$time)
      y_min <- min(df$conc); y_max <- max(df$conc)

      ann$x_pos <- c(x_min + 0.05 * (x_max - x_min),
                     x_max - 0.05 * (x_max - x_min))[seq_len(nrow(ann))]
      ann$y_pos <- c(y_min * 0.5, y_max / 0.5)[seq_len(nrow(ann))]
      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(0, 1)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    }

    print(p)
  }

  list(
    fitted_parameters = fit_results,
    data = df,
    plot = if (plot) p else NULL
  )
}
