#' Screen number of compartments
#'
#' Runs candidate models with one, two, and three compartments by modifying only
#' the compartment setting in the current model code.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the folder name for storing
#'   intermediate results. If \code{NULL} (default), \code{tempdir()}
#'   is used for temporary storage. If specified, a cache directory
#'   is created in the current working directory.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Internal environment used to store model indices and cached
#'   results across steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to mod.run. These may include
#'   custom_base, which is used to initialize the baseline model when no
#'   best_code is present in start.mod.
#'
#' @details
#' Three candidate models are created by modifying only the number of
#' compartments in the starting model code. The candidate codes are evaluated
#' sequentially, and a results table containing model names, model codes,
#' Fitness values, and information criteria is returned for logging and
#' decision making.
#'
#' @return A list with the following elements:
#' \itemize{
#'   \item results_table: a data frame with one row per candidate model,
#'         including model description and fit statistics
#'   \item best_code: named integer vector corresponding to the best candidate
#'   \item best_row: one-row data frame containing the best candidate summary
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#'   string <- c(1, 0, 0, 0, 0, 0, 0, 0, 0, 1)
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#'   penalty.control = penaltyControl()
#'   penalty.control$penalty.terms = c("rse", "theta", "covariance")
#'   step_compartments(
#'     dat = dat,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_cmpt_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export

step_compartments <-
  function(dat,
           start.mod = NULL,
           search.space = "ivbase",
           no.cores = NULL,
           param_table = NULL,
           penalty.control = NULL,
           precomputed_results_file = NULL,
           filename = "test",
           foldername = NULL,
           .modEnv = NULL,
           verbose = TRUE,
           ...) {
    # Determine which .modEnv to use
    if (!is.null(.modEnv)) {
      if (!is.environment(.modEnv)) {
        stop("`.modEnv` must be an environment", call. = FALSE)
      }
    } else {
      .modEnv <- new.env(parent = emptyenv())
    }

    # Ensure essential keys exist in .modEnv
    if (is.null(.modEnv$modi))
      .modEnv$modi <- 1L
    if (is.null(.modEnv$r))
      .modEnv$r <- 1L
    if (is.null(.modEnv$Store.all))
      .modEnv$Store.all <- NULL
    if (is.null(.modEnv$precomputed_cache_loaded))
      .modEnv$precomputed_cache_loaded <- FALSE
    if (is.null(.modEnv$precomputed_results))
      .modEnv$precomputed_results <- NULL
    if (is.null(.modEnv$param_table))
      .modEnv$param_table <- NULL

    if (is.null(no.cores)) {
      no.cores <- rxode2::getRxThreads()
    }

    if (!is.null(start.mod)) {
      current_code <- start.mod
    } else {
      current_code <-
        base_model(search.space = search.space)
    }

    candidate_codes <- lapply(1:3, function(k) {
      code <- current_code
      code["no.cmpt"] <- k
      code
    })

    fits <- vapply(candidate_codes, function(code_vec) {
      mod.run(
        string       = unname(code_vec),
        dat          = dat,
        search.space = search.space,
        no.cores = no.cores,
        param_table  = param_table,
        precomputed_results_file =   precomputed_results_file,
        filename = filename,
        foldername = foldername,
        .modEnv = .modEnv,
        verbose = verbose,
        ...
      )
    }, numeric(1))

    model_names <- vapply(candidate_codes,
                          function(code)
                            parseName(modcode = unname(code), search.space = search.space),
                          character(1))
    model_codes_chr <- vapply(candidate_codes,
                              function(code)
                                paste(unname(code), collapse = ","),
                              character(1))

    AICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$AIC
    BICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$BIC
    OFVvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$OBJFV

    results_table <- data.frame(
      Step          = "No. of compartments",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name"  = model_names,
      "Model code"  = model_codes_chr,
      Fitness       = fits,
      AIC           = c(AICvals),
      BIC           = c(BICvals),
      OFV           = c(OFVvals),
      stringsAsFactors = FALSE
    )

    best_idx  <- which.min(results_table$Fitness)
    best_row  <- results_table[best_idx, , drop = FALSE]
    best_code <- candidate_codes[[best_idx]]

    # Increment round number
    .modEnv$r <- .modEnv$r + 1L

    list(results_table = results_table,
         best_code     = best_code,
         best_row      = best_row)
  }


#' Screen elimination type (linear vs Michaelis-Menten)
#'
#' Runs linear and Michaelis-Menten elimination candidates by modifying only the
#' elimination setting in the current model code.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional internal environment used to store model indices
#'   and cached results across model-selection steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to \code{mod.run()}.
#'
#' @details
#' When mm = 0, any inter-individual variability term for Km
#' (eta.km) present in the model code is automatically set to zero.
#'
#' @return A list with the following elements:
#' \itemize{
#'   \item results_table: a data.frame with one row per candidate
#'   model, including model description, Fitness, AIC, BIC, and OFV.
#'   \item best_code: named integer vector corresponding to the best
#'   candidate's model code.
#'   \item best_row: one-row data.frame summarizing the best
#'   candidate.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#'   penalty.control = penaltyControl()
#'   penalty.control$penalty.terms = c("rse", "theta", "covariance")
#'   # Initialize start.mod with a base model
#'    start.mod <- base_model("ivbase")
#'   step_elimination(
#'     dat = dat,
#'     start.mod = start.mod,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_elim_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export

step_elimination <-
  function(dat,
           start.mod = NULL,
           search.space = "ivbase",
           no.cores = NULL,
           param_table = NULL,
           penalty.control = NULL,
           precomputed_results_file = NULL,
           filename = "test",
           foldername = NULL,
           .modEnv = NULL,
           verbose = TRUE,
           ...) {
    # Determine which .modEnv to use
    if (!is.null(.modEnv)) {
      if (!is.environment(.modEnv)) {
        stop("`.modEnv` must be an environment", call. = FALSE)
      }
      # .modEnv <- get(".modEnv", inherits = TRUE)
    } else {
      .modEnv <- new.env(parent = emptyenv())
    }

    # Ensure essential keys exist in .modEnv
    if (is.null(.modEnv$modi))
      .modEnv$modi <- 1L
    if (is.null(.modEnv$r))
      .modEnv$r <- 1L
    if (is.null(.modEnv$Store.all))
      .modEnv$Store.all <- NULL
    if (is.null(.modEnv$precomputed_cache_loaded))
      .modEnv$precomputed_cache_loaded <- FALSE
    if (is.null(.modEnv$precomputed_results))
      .modEnv$precomputed_results <- NULL
    if (is.null(.modEnv$param_table))
      .modEnv$param_table <- NULL

    if (is.null(no.cores)) {
      no.cores <- rxode2::getRxThreads()
    }
    # starting code
    if (!is.null(start.mod)) {
      current_code <- start.mod
    } else {
      current_code <-
        base_model(search.space = search.space)
    }

    # candidates: mm = 0 and mm = 1
    candidate_codes <- lapply(c(0L, 1L), function(m) {
      code <- current_code
      code["mm"] <- m
      # Optional: if no MM, Km IIV should be off
      if (m == 0L && "eta.km" %in% names(code))
        code["eta.km"] <- 0L
      code
    })

    fits <- vapply(candidate_codes, function(code_vec) {
      mod.run(
        string       = unname(code_vec),
        dat          = dat,
        search.space = search.space,
        no.cores = no.cores,
        param_table  = param_table,
        precomputed_results_file =   precomputed_results_file,
        filename = filename,
        foldername = foldername,
        .modEnv = .modEnv,
        verbose = verbose,
        ...
      )
    }, numeric(1))

    model_names <- vapply(candidate_codes,
                          function(code)
                            parseName(modcode = unname(code),search.space = search.space),
                          character(1))
    model_codes_chr <-
      vapply(candidate_codes, function(code)
        paste(unname(code), collapse = ","), character(1))

    AICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$AIC
    BICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$BIC
    OFVvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$OBJFV

    results_table <- data.frame(
      Step        = "Elimination type",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name" = model_names,
      "Model code" = model_codes_chr,
      Fitness      = fits,
      AIC           = c(AICvals),
      BIC           = c(BICvals),
      OFV           = c(OFVvals),
      stringsAsFactors = FALSE
    )

    best_idx <- which.min(results_table$Fitness)
    best_row  <- results_table[best_idx, , drop = FALSE]
    best_code <- candidate_codes[[best_idx]]

    # Increment round number
    .modEnv$r <- .modEnv$r + 1L

    list(results_table = results_table,
         best_code     = best_code,
         best_row      = best_row)
  }


#' Evaluate inter-individual variability on Km
#'
#' Runs candidate models with and without IIV on \eqn{K_m} by modifying only the
#' corresponding random-effect setting in the current model code.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase". Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional internal environment used to store model indices
#'   and cached results across model-selection steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments forwarded to mod.run().
#'
#' @details
#' This step is executed only when the starting model code specifies
#' Michaelis--Menten elimination (mm = 1). If mm is not equal to 1 in
#' the starting model, no model comparison is performed.

#' @return
#' A list with the following elements:
#' \itemize{
#'   \item results_table: A data.frame summarizing the evaluated models.
#'   \item best_code: A named integer vector representing the selected
#'     model code.
#'   \item best_row: A one-row data.frame corresponding to the selected
#'     model.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#'   penalty.control <- penaltyControl()
#'   penalty.control$penalty.terms <-
#'     c("rse", "theta", "covariance", "shrinkage", "omega")
#'   start.mod <- base_model("ivbase")
#'   start.mod["mm"] <- 1L
#'   step_iiv_km(
#'     dat = dat,
#'     start.mod = start.mod,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_etakm_test",
#'     penalty.control = penalty.control,
#'    saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export

step_iiv_km <- function(dat,
                        start.mod = NULL,
                        search.space ="ivbase",
                        no.cores = NULL,
                        param_table = NULL,
                        penalty.control = NULL,
                        precomputed_results_file = NULL,
                        filename = "test",
                        foldername = NULL,
                        .modEnv = NULL,
                        verbose = TRUE,
                        ...) {
  # Determine which .modEnv to use
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  # Starting code
  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }

  # eta.km (only if mm = 1)
  if ("mm" %in% names(current_code) && current_code["mm"] == 1L &&
      "eta.km" %in% names(current_code) &&
      current_code["eta.km"] == 0L) {
    candidate_codes <- list(current_code,
                            {
                              tmp <- current_code
                              tmp["eta.km"] <- 1L
                              tmp
                            })

    fits <- vapply(candidate_codes, function(code_vec) {
      mod.run(
        string       = unname(code_vec),
        dat          = dat,
        search.space = search.space,
        no.cores = no.cores,
        param_table  = param_table,
        precomputed_results_file =   precomputed_results_file,
        filename = filename,
        foldername = foldername,
        .modEnv = .modEnv,
        verbose = verbose,
        ...
      )
    }, numeric(1))

    model_names <- vapply(candidate_codes,
                          function(code)
                            parseName(modcode = unname(code), search.space = search.space),
                          character(1))

    model_codes_chr <- vapply(candidate_codes, function(code) {
      paste(unname(code), collapse = ",")
    }, character(1))

    AICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$AIC
    BICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$BIC
    OFVvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$OBJFV

    results_table <- data.frame(
      Step        = "IIV on Km",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name" = model_names,
      "Model code" = model_codes_chr,
      Fitness      = fits,
      AIC           = c(AICvals),
      BIC           = c(BICvals),
      OFV           = c(OFVvals),
      stringsAsFactors = FALSE
    )

    best_idx <- which.min(results_table$Fitness)
    best_row  <- results_table[best_idx, , drop = FALSE]
    best_code <- candidate_codes[[best_idx]]

    # Increment round number
    .modEnv$r <- .modEnv$r + 1L

    list(
      results_table = results_table,
      best_code     = best_code,
      best_row      = best_row
    )
  }
}


#' Evaluate inter-individual variability on Ka
#'
#' Runs candidate models with and without IIV on \eqn{K_a} by modifying only the
#' corresponding random-effect setting in the current model code.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase". Default is "oralbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a temporary path is used via \code{tempdir()}.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv An optional environment used to store intermediate
#'   results across model runs.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments forwarded to `mod.run()`.
#'
#' @details
#' This step is executed only when the search space is "oralbase" and the
#' starting model code does not already include inter-individual
#' variability on \eqn{K_a}. If these conditions are not met, no model
#' comparison is performed.
#'
#' @return
#' A list with the following elements:
#' \itemize{
#'   \item results_table: A data.frame summarizing the evaluated models.
#'   \item best_code: A named integer vector representing the selected
#'     model code.
#'   \item best_row: A one-row data.frame corresponding to the selected
#'     model.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- theo_sd
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(2)
#'   param_table$init[param_table$Name == "lvc"] <- log(30)
#'   penalty.control <- penaltyControl()
#'   penalty.control$penalty.terms <-
#'     c("rse", "theta", "covariance", "shrinkage", "omega")
#'   start.mod <- base_model("oralbase")
#'   step_iiv_ka(
#'     dat = dat,
#'     start.mod = start.mod,
#'     search.space = "oralbase",
#'     param_table = param_table,
#'     filename = "step_etaka_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export
#'
step_iiv_ka <- function(dat,
                        start.mod = NULL,
                        search.space = "oralbase",
                        no.cores = NULL,
                        param_table = NULL,
                        penalty.control = NULL,
                        precomputed_results_file = NULL,
                        filename = "test",
                        foldername = NULL,
                        .modEnv = NULL,
                        verbose = TRUE,
                        ...) {
  # Determine which .modEnv to use
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  # Starting code
  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }

  # eta.ka (only for oralbase)
  if (identical(search.space, "oralbase") &&
      "eta.ka" %in% names(current_code) &&
      current_code["eta.ka"] == 0L) {
    candidate_codes <- list(current_code,
                            {
                              tmp <- current_code
                              tmp["eta.ka"] <- 1L
                              tmp
                            })

    fits <- vapply(candidate_codes, function(code_vec) {
      mod.run(
        string       = unname(code_vec),
        dat          = dat,
        search.space = search.space,
        no.cores = no.cores,
        param_table  = param_table,
        precomputed_results_file =   precomputed_results_file,
        filename = filename,
        foldername = foldername,
        .modEnv = .modEnv,
        verbose = verbose,
        ...
      )
    }, numeric(1))

    model_names <- vapply(candidate_codes,
                          function(code)
                            parseName(modcode = unname(code), search.space = search.space),
                          character(1))

    model_codes_chr <- vapply(candidate_codes, function(code) {
      paste(unname(code), collapse = ",")
    }, character(1))

    AICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$AIC
    BICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$BIC
    OFVvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$OBJFV

    results_table <- data.frame(
      Step        = "IIV on Ka",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name" = model_names,
      "Model code" = model_codes_chr,
      Fitness      = fits,
      AIC           = c(AICvals),
      BIC           = c(BICvals),
      OFV           = c(OFVvals),
      stringsAsFactors = FALSE
    )

    best_idx <- which.min(results_table$Fitness)
    best_row  <- results_table[best_idx, , drop = FALSE]
    best_code <- candidate_codes[[best_idx]]

    # Increment round number
    .modEnv$r <- .modEnv$r + 1L

    list(
      results_table = results_table,
      best_code     = best_code,
      best_row      = best_row
    )
  }
}


#' Forward selection of IIV on structural parameters
#'
#' Implements a forward selection procedure to assess the inclusion of
#' inter-individual variability on structural pharmacokinetic parameters.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional environment for storing intermediate results
#'   across model runs.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to the model estimation
#'   function.
#'
#' @details
#' The procedure begins with an initial model and proceeds iteratively.
#' At each step, candidate models are generated by adding exactly one
#' additional IIV (random-effect) term while keeping all other aspects of
#' the model unchanged. If any candidate improves the chosen fitness
#' criterion, the best-improving candidate becomes the new reference model
#' for the next iteration. The algorithm stops when no further improvement
#' is achieved.
#' The set of parameters eligible for IIV depends on the number of
#' compartments:
#' \itemize{
#'   \item One-compartment models: clearance and central volume
#'   \item Two-compartment models: clearance, central volume, peripheral
#'         volume, and inter-compartmental clearance
#'   \item Three-compartment models: clearance, central volume,
#'         peripheral volumes, and inter-compartmental clearances
#' }
#'
#' @return
#' A list with three elements:
#' \itemize{
#'   \item results_table: A data frame summarizing all models evaluated
#'     during the forward selection process.
#'   \item best_code: A named integer vector corresponding to the selected
#'     model.
#'   \item best_row: A one-row data frame containing the results of the
#'     selected model.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- Bolus_2CPT[Bolus_2CPT$SD==1,]
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(4)
#'   param_table$init[param_table$Name == "lvc2cmpt"] <- log(70)
#'   param_table$init[param_table$Name == "lvp2cmpt"] <- log(40)
#'   param_table$init[param_table$Name == "lq2cmpt"] <- log(4)
#'   penalty.control <- penaltyControl()
#'   penalty.control$penalty.terms <-
#'     c("rse", "theta", "covariance", "shrinkage", "omega")
#'   start.mod <- base_model("ivbase")
#'   start.mod["no.cmpt"] <- 2L
#'   step_iiv_f(
#'     dat = dat,
#'     start.mod = start.mod,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_eta_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export
#'
#'
step_iiv_f <- function(dat,
                       start.mod = NULL,
                       search.space = "ivbase",
                       no.cores = NULL,
                       param_table = NULL,
                       penalty.control = NULL,
                       precomputed_results_file = NULL,
                       filename = "test",
                       foldername = NULL,
                       .modEnv = NULL,
                       verbose = TRUE,
                       ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  # Starting code
  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }
  # Structural IIV (forward selection)
  struct_iiv <- c("eta.vc")

  no.cmpt <- suppressWarnings(as.integer(current_code[["no.cmpt"]]))
  if (no.cmpt >= 2L) {
    struct_iiv <- c(struct_iiv, "eta.vp", "eta.q")
  }
  if (no.cmpt >= 3L) {
    struct_iiv <- c(struct_iiv, "eta.vp2", "eta.q2")
  }
  # struct_iiv <- struct_iiv[struct_iiv %in% names(current_code)]

  keep_going <- TRUE
  all_results <- NULL
  while (keep_going) {
    available <- struct_iiv[current_code[struct_iiv] == 0L]
    if (length(available) == 0L)
      break
    # Baseline + "add one IIV" candidates
    candidate_codes <- vector("list", length(available) + 1L)
    candidate_codes[[1L]] <- current_code
    for (i in seq_along(available)) {
      nm <- available[i]
      tmp <- current_code
      tmp[nm] <- 1L
      candidate_codes[[i + 1L]] <- tmp
    }

    fits <- vapply(candidate_codes, function(code_vec) {
      mod.run(
        string       = unname(code_vec),
        dat          = dat,
        search.space = search.space,
        no.cores = no.cores,
        param_table  = param_table,
        precomputed_results_file =   precomputed_results_file,
        filename = filename,
        foldername = foldername,
        .modEnv = .modEnv,
        verbose = verbose,
        ...
      )
    }, numeric(1))

    model_names <- vapply(candidate_codes,
                          function(code)
                            parseName(modcode = unname(code), search.space = search.space),
                          character(1))

    model_codes_chr <- vapply(candidate_codes, function(code) {
      paste(unname(code), collapse = ",")
    }, character(1))

    AICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$AIC
    BICvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$BIC
    OFVvals <-
      .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r, ]$OBJFV

    results_table <- data.frame(
      Step        = "IIV (forward)",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name" = model_names,
      "Model code" = model_codes_chr,
      Fitness      = fits,
      AIC           = c(AICvals),
      BIC           = c(BICvals),
      OFV           = c(OFVvals),
      stringsAsFactors = FALSE
    )

    all_results <- rbind(all_results, results_table)

    .modEnv$r <- .modEnv$r + 1L

    best_idx <- which.min(results_table$Fitness)

    # If baseline remains best (index 1), stop; else accept improvement and continue
    if (best_idx == 1L) {
      keep_going <- FALSE
    } else {
      current_code <- candidate_codes[[best_idx]]
    }
  }

  if (!is.null(all_results)) {
    best_row <-
      all_results[which.min(all_results$Fitness), , drop = FALSE]
    best_idx <- which.min(results_table$Fitness)
    best_code <- candidate_codes[[best_idx]]
  } else {
    # return the original start model
    base_fit <- mod.run(
      string           = unname(current_code),
      dat              = dat,
      search.space     = search.space,
      no.cores = no.cores,
      param_table      = param_table,
      penalty.control  = penalty.control,
      precomputed_results_file =   precomputed_results_file,
      foldername = foldername,
      filename = filename,
      .modEnv = .modEnv,
      verbose = verbose,
      ...
    )

    last_row <- utils::tail(.modEnv$Store.all, 1)
    best_row <- data.frame(
      Step = "IIV (forward)",
      "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
      "Model name" =
        parseName(modcode = unname(current_code), search.space = search.space),
      "Model code" = paste(unname(current_code), collapse = ","),
      Fitness = base_fit,
      AIC     = last_row$AIC,
      BIC     = last_row$BIC,
      OFV     = last_row$OBJFV,
      stringsAsFactors = FALSE
    )
    all_results <- best_row
    best_code <- current_code

    .modEnv$r <- .modEnv$r + 1L
  }

  list(results_table = all_results,
       best_code     = current_code,
       best_row      = best_row)
}


#' Evaluate inclusion of ETA correlation structure
#'
#' Evaluates whether correlation between inter-individual random
#' effects (ETA correlation) should be included in the model.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param penalty.control A list of penalty control parameters defined by
#'   \code{penaltyControl()}, specifying penalty values used for model diagnostics
#'   during fitness evaluation.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional environment used to store model indices and cached
#'   results across steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to the model estimation function.
#'
#' @details
#' Two candidate models are constructed by toggling the correlation setting
#' of inter-individual random effects in the model code. Model selection is
#' based on comparison of Fitness values returned during estimation.
#'
#' @return A list with the following elements:
#' \itemize{
#'   \item results_table: A data frame summarizing the evaluated models,
#'   \item best_code: A named integer vector corresponding to the selected
#'     model code,
#'   \item best_row: A one-row data frame containing the summary of the
#'     selected model.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#'   penalty.control <- penaltyControl()
#'   penalty.control$penalty.terms <-
#'     c("rse", "theta", "covariance", "shrinkage", "omega")
#'   start.mod <- base_model("ivbase")
#'   start.mod["eta.vc"] <- 1L
#'   step_correlation(
#'     dat = dat,
#'     start.mod = start.mod,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_mcorr_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export
#'
step_correlation <- function(dat,
                             start.mod = NULL,
                             search.space = "ivbase",
                             no.cores = NULL,
                             param_table = NULL,
                             penalty.control = NULL,
                             precomputed_results_file = NULL,
                             filename = "test",
                             foldername = NULL,
                             .modEnv = NULL,
                             verbose = TRUE,
                             ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  # starting code
  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }

  corrcode <- current_code["mcorr"]
  alt_val <- 1L - corrcode

  candidate_codes <- list(current_code,
                          {
                            tmp <- current_code
                            tmp["mcorr"] <- alt_val
                            tmp
                          })

  fits <- vapply(candidate_codes, function(code_vec) {
    mod.run(
      string       = unname(code_vec),
      dat          = dat,
      search.space = search.space,
      no.cores = no.cores,
      param_table  = param_table,
      precomputed_results_file =   precomputed_results_file,
      filename = filename,
      foldername = foldername,
      .modEnv = .modEnv,
      verbose =  verbose,
      ...
    )
  }, numeric(1))

  model_names <- vapply(candidate_codes,
                        function(code)
                          parseName(modcode = unname(code), search.space = search.space),
                        character(1))

  model_codes_chr <- vapply(candidate_codes, function(code) {
    paste(unname(code), collapse = ",")
  }, character(1))

  AICvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$AIC
  BICvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$BIC
  OFVvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$OBJFV

  results_table <- data.frame(
    Step        = "Eta correlation",
    "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
    "Model name" = model_names,
    "Model code" = model_codes_chr,
    Fitness      = fits,
    AIC           = c(AICvals),
    BIC           = c(BICvals),
    OFV           = c(OFVvals),
    stringsAsFactors = FALSE
  )

  best_idx <- which.min(results_table$Fitness)
  best_row  <- results_table[best_idx, , drop = FALSE]
  best_code <- candidate_codes[[best_idx]]

  .modEnv$r <- .modEnv$r + 1L

  list(results_table = results_table,
       best_code     = best_code,
       best_row      = best_row)
}


#' Evaluate residual error model structure
#'
#' Evaluates alternative residual error model structures by modifying the
#' residual variability setting in the model code.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of \code{ivbase} or \code{oralbase}.
#'   Default is \code{ivbase}.
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param param_table Optional parameter table used during model estimation.
#' @param penalty.control Optional penalty control object used for reporting
#'   penalty terms in the results table.
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional environment used to store model indices and cached
#'   results across steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to the model estimation function.
#'
#' @details
#' Candidate models are constructed by assigning different residual error
#' types to the model code. Each candidate differs only in the residual
#' variability specification, and all other structural and statistical
#' components are kept unchanged. Model selection is based on comparison
#' of Fitness values obtained during estimation.
#'
#' @return A list with the following elements:
#' \itemize{
#'   \item results_table: A data frame summarizing the evaluated residual
#'     error models and their fit statistics,
#'   \item best_code: A named integer vector corresponding to the selected
#'     model code,
#'   \item best_row: A one-row data frame containing the summary of the
#'     selected model.
#' }
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#'   dat <- pheno_sd
#'   param_table <- initialize_param_table()
#'   param_table$init[param_table$Name == "lcl"] <- log(0.008)
#'   param_table$init[param_table$Name == "lvc"] <- log(0.6)
#'   penalty.control <- penaltyControl()
#'   penalty.control$penalty.terms <-
#'     c("rse","theta", "covariance","shrinkage","omega","correlation","sigma")
#'   step_rv(
#'     dat = dat,
#'     search.space = "ivbase",
#'     param_table = param_table,
#'     filename = "step_rv_test",
#'     penalty.control = penalty.control,
#'     saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=15,nEm=15)
#'   )
#' }
#'
#' @seealso \code{\link{mod.run}}, \code{\link{base_model}}, \code{\link{penaltyControl}}
#'
#' @export
step_rv <- function(dat,
                    start.mod = NULL,
                    search.space = "ivbase",
                    no.cores = NULL,
                    param_table = NULL,
                    penalty.control = NULL,
                    precomputed_results_file = NULL,
                    filename = "test",
                    foldername = NULL,
                    .modEnv = NULL,
                    verbose = TRUE,
                    ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL

  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  # starting code
  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }

  # baseline candidate
  candidate_codes <- list(current_code)

  candidate_codes <- lapply(1:3, function(rv_val) {
    code <- current_code
    code["rv"] <- rv_val
    code
  })

  fits <- vapply(candidate_codes, function(code_vec) {
    mod.run(
      string       = unname(code_vec),
      dat          = dat,
      search.space = search.space,
      no.cores = no.cores,
      param_table  = param_table,
      precomputed_results_file =   precomputed_results_file,
      filename = filename,
      foldername = foldername,
      .modEnv = .modEnv,
      verbose = verbose,
      ...
    )
  }, numeric(1))

  model_names <- vapply(candidate_codes,
                        function(code)
                          parseName(modcode = unname(code), search.space = search.space),
                        character(1))

  model_codes_chr <- vapply(candidate_codes, function(code) {
    paste(unname(code), collapse = ",")
  }, character(1))

  AICvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$AIC
  BICvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$BIC
  OFVvals <-
    .modEnv$Store.all[.modEnv$Store.all$round.num == .modEnv$r,]$OBJFV

  results_table <- data.frame(
    Step        = "Residual error types",
    "Penalty terms" = paste(penalty.control$penalty.terms, collapse = ", "),
    "Model name" = model_names,
    "Model code" = model_codes_chr,
    Fitness      = fits,
    AIC           = c(AICvals),
    BIC           = c(BICvals),
    OFV           = c(OFVvals),
    stringsAsFactors = FALSE
  )

  best_idx <- which.min(results_table$Fitness)
  best_row  <- results_table[best_idx, , drop = FALSE]
  best_code <- candidate_codes[[best_idx]]

  .modEnv$r <- .modEnv$r + 1L

  list(results_table =  results_table,
       best_code     =  best_code,
       best_row      =  best_row)
}



#' Stepwise model building operator for model selection
#'
#' Implements automated stepwise model selection for structural and statistical
#' components of nonlinear mixed-effects models, evaluating the number of
#' compartments, elimination type, inter-individual variability, correlation
#' structures, and residual error models.
#'
#' @param dat A data frame containing pharmacokinetic data in standard
#'   nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
#'   additional columns.
#' @param param_table Optional data frame of initial parameter estimates. If NULL,
#'   the table is generated by \code{auto_param_table()}.
#' @param start.mod A named integer vector specifying the starting model
#'   code. If NULL, a base model is generated using \code{base_model()}.
#' @param search.space Character, one of "ivbase" or "oralbase".
#'   Default is "ivbase".
#' @param no.cores Integer. Number of CPU cores to use. If NULL, uses
#'   \code{rxode2::getRxThreads()}.
#' @param penalty.control An object created by \code{penaltyControl()} defining
#'   penalty terms used in the fitness calculation.
#' @param dynamic_fitness Logical; if `TRUE`, the set of penalty terms may
#'   change dynamically across steps.
#' @param steps Numeric or character vector defining the sequence of steps
#'   to be executed. Each digit corresponds to a specific step:
#'   \describe{
#'     \item{1}{Number of compartments}
#'     \item{2}{Elimination type}
#'     \item{3}{IIV on Km}
#'     \item{4}{IIV on Ka}
#'     \item{5}{Forward selection of structural IIV}
#'     \item{6}{Correlation between random effects}
#'     \item{7}{Residual error model}
#'   }
#' @param precomputed_results_file Optional path to a CSV file of previously computed
#'   model results used for caching.
#' @param foldername Character string specifying the name of the folder to be
#'   created in the current working directory to store intermediate results.
#'   If NULL, a name is generated automatically.
#' @param filename Optional character string used as a prefix for output files.
#'   Defaults to "test".
#' @param .modEnv Optional environment used internally to store model indices,
#'   cached parameter tables, and results across steps.
#' @param verbose Logical. If TRUE, print progress messages.
#' @param ... Additional arguments passed to \code{mod.run()}.
#'
#' @details
#' The stepwise procedure iterates over the specified steps in order.
#' At each step, only a single component of the model is modified, while
#' all other structural and statistical elements remain unchanged.
#' Model comparison is based on a scalar fitness criterion returned by
#' the estimation routine.
#'
#' The order and inclusion of steps are controlled by the user via a
#' numeric step code sequence. Steps that are not applicable to the
#' current model configuration may be skipped automatically.
#'
#' The final best model is defined as the model with the minimum fitness
#' value in the last completed estimation round.
#'
#' @return
#' An object of class \code{"sfOperatorResult"} with the following elements:
#' \itemize{
#'   \item \code{"Final Best Code"}: Named integer vector of the selected model code.
#'   \item \code{"Final Best Model Name"}: Character string identifying the best model.
#'   \item \code{"Stepwise Best Models"}: Data frame summarizing the best model
#'     selected at each executed step.
#'   \item \code{"Stepwise History"}: Named list containing full results for
#'     each step using descriptive step names.
#'   \item \code{"Model Run History"}: Data frame containing all model runs
#'     performed during the procedure.
#' }
#'
#' @seealso
#' \code{\link{step_compartments}},
#' \code{\link{step_elimination}},
#' \code{\link{step_iiv_km}},
#' \code{\link{step_iiv_f}},
#' \code{\link{step_correlation}},
#' \code{\link{step_rv}}
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#' out<-sf.operator(
#'   dat = pheno_sd,
#'   steps = 1234,
#'   search.space = "ivbase",
#'   saem.control = nlmixr2est::saemControl(
#'     seed = 1234,
#'     nBurn = 200,
#'     nEm   = 300,
#'     logLik = TRUE
#'   )
#' )
#' print(out)
#' }
#'
#' @seealso
#' \code{\link{auto_param_table}}, \code{\link{base_model}},
#' \code{\link{penaltyControl}}, \code{\link{mod.run}}, \code{\link{ppkmodGen}},
#' \code{\link{step_compartments}}, \code{\link{step_elimination}},
#' \code{\link{step_iiv_km}}, \code{\link{step_iiv_ka}}, \code{\link{step_iiv_f}},
#' \code{\link{step_correlation}}, \code{\link{step_rv}}
#'
#' @export

sf.operator <- function(dat,
                        start.mod = NULL,
                        search.space = "ivbase",
                        no.cores = NULL,
                        param_table = NULL,
                        steps = 123567,
                        dynamic_fitness = TRUE,
                        penalty.control = penaltyControl(),
                        precomputed_results_file = NULL,
                        foldername = NULL,
                        filename = "test",
                        .modEnv = NULL,
                        verbose = TRUE,
                        ...) {
  if (!is.null(.modEnv)) {
    if (!is.environment(.modEnv)) {
      stop("`.modEnv` must be an environment", call. = FALSE)
    }
    # .modEnv <- get(".modEnv", inherits = TRUE)
  } else {
    .modEnv <- new.env(parent = emptyenv())
  }

  # Ensure essential keys exist in .modEnv
  if (is.null(.modEnv$modi))
    .modEnv$modi <- 1L
  if (is.null(.modEnv$r))
    .modEnv$r <- 1L
  if (is.null(.modEnv$Store.all))
    .modEnv$Store.all <- NULL
  if (is.null(.modEnv$precomputed_cache_loaded))
    .modEnv$precomputed_cache_loaded <- FALSE
  if (is.null(.modEnv$precomputed_results))
    .modEnv$precomputed_results <- NULL
  if (is.null(.modEnv$param_table))
    .modEnv$param_table <- NULL
  if (is.null(.modEnv$saem.control))
    .modEnv$saem.control <- NULL
  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }
  if (is.null(no.cores)) {
    no.cores <- rxode2::getRxThreads()
  }

  if (is.null(foldername) || !nzchar(foldername)) {
    # foldername <- paste0("stepCache_", filename, "_", digest::digest(dat))
    foldername <- tempdir()
  }
  if (!dir.exists(foldername)) {
    dir.create(foldername, showWarnings = FALSE, recursive = TRUE)
  }
  # step map
  # .step_map <- list(
  #   `1` = step_compartments,
  #   `2` = step_elimination,
  #   `3` = step_iiv_km,
  #   `4` = step_iiv_ka,
  #   `5` = step_iiv_f,
  #   `6` = step_correlation,
  #   `7` = step_rv
  # )
  .step_map <- list(
    `1` = get("step_compartments", envir = parent.frame()),
    `2` = get("step_elimination",  envir = parent.frame()),
    `3` = get("step_iiv_km",        envir = parent.frame()),
    `4` = get("step_iiv_ka",        envir = parent.frame()),
    `5` = get("step_iiv_f",         envir = parent.frame()),
    `6` = get("step_correlation",   envir = parent.frame()),
    `7` = get("step_rv",            envir = parent.frame())
  )

  .step_name <- c(
    `1` = "Number of compartments",
    `2` = "Elimination type",
    `3` = "IIV on Km",
    `4` = "IIV on Ka",
    `5` = "IIV (forward selection)",
    `6` = "ETA correlation",
    `7` = "Residual error model"
  )

  .penalty_map <- list(
    `1` = c("rse", "theta", "covariance"),
    `2` = c("rse", "theta", "covariance"),
    `3` = c("rse", "theta", "covariance", "shrinkage", "omega"),
    `4` = c("rse", "theta", "covariance", "shrinkage", "omega"),
    `5` = c("rse", "theta", "covariance", "shrinkage", "omega"),
    `6` = c(
      "rse",
      "theta",
      "covariance",
      "shrinkage",
      "omega",
      "correlation"
    ),
    `7` = c(
      "rse",
      "theta",
      "covariance",
      "shrinkage",
      "omega",
      "correlation",
      "sigma"
    )
  )

  # Parse steps (allow integer or character)
  if (length(steps) == 1) {
    steps <- strsplit(as.character(steps), "")[[1]]
  }

  # Validate steps
  bad_steps <- setdiff(steps, names(.step_map))
  if (length(bad_steps) > 0) {
    stop("Unknown step codes: ",
         paste(bad_steps, collapse = ", "),
         call. = FALSE)
  }

  # Initial estimates
  if (!is.null(param_table)) {
    param_table_use <- param_table
  } else if (!is.null(.modEnv$param_table)) {
    param_table_use <- .modEnv$param_table
  } else {
    param_table_use <- auto_param_table(
      dat = dat,
      nlmixr2autoinits = TRUE,
      foldername = foldername,
      filename = filename,
      out.inits = TRUE
    )
    .modEnv$param_table <- param_table_use
  }

  param_table <- param_table_use

  if (!is.null(start.mod)) {
    current_code <- start.mod
  } else {
    current_code <-
      base_model(search.space = search.space)
  }

  step_results <- list()
  for (s in steps) {
    # dynamic fitness control
    if (isTRUE(dynamic_fitness)) {
      new_terms <- .penalty_map[[as.character(s)]]
      if (!is.null(new_terms)) {
        penalty.control$penalty.terms <- new_terms
      }
    }

    step_fun <- .step_map[[s]]
    if (verbose) {
      message(crayon::blue(
        paste0(
          "Running Step ",
          s,
          ": ",
          .step_name[[as.character(s)]],
          " ----------------------------------------------------"
        )
      ))
    }

    res <- step_fun(
      dat = dat,
      start.mod = current_code,
      search.space = search.space,
      param_table = param_table,
      penalty.control = penalty.control,
      precomputed_results_file = precomputed_results_file,
      filename = filename,
      foldername = foldername,
      .modEnv = .modEnv,
      ...
    )

    # Some steps are conditional and may return NULL
    if (is.null(res)) {
      if (verbose) {
        message(paste0("Step ",
                       s,
                       ": ",
                       .step_name[[as.character(s)]],
                       " skipped."))
      }
      next
    }

    step_results[[s]] <- res
    current_code <- res$best_code
  }

  out <- list()
  class(out) <- "sfOperatorResult"
  latest_round <-
    subset(.modEnv$Store.all,
           round.num == max(.modEnv$Store.all$round.num, na.rm = TRUE))
  best_row <- latest_round[which.min(latest_round$fitness), ]

  if (search.space == "ivbase") {
    cols_to_extract <- c(
      "no.cmpt",
      "eta.vmax",
      "eta.km",
      "eta.cl",
      "eta.vc",
      "eta.vp",
      "eta.vp2",
      "eta.q",
      "eta.q2",
      "mm",
      "mcorr",
      "rv"
    )
  } else {
    cols_to_extract <- c(
      "no.cmpt",
      "eta.vmax",
      "eta.km",
      "eta.cl",
      "eta.vc",
      "eta.vp",
      "eta.vp2",
      "eta.q",
      "eta.q2",
      "eta.ka",
      "mm",
      "mcorr",
      "rv"
    )
  }

  # Assign the selected columns from the best row to the output
  tmp <- best_row[, cols_to_extract, drop = FALSE]
  rownames(tmp) <- NULL
  out[["Final Best Code"]] <- tmp

  last_step <- utils::tail(step_results, 1)[[1]]

  out[["Final Best Model Name"]] <-last_step$best_row$Model.name

  out[["Stepwise Best Models"]] <-
    do.call(rbind,
            Map(function(s, x) {
              df <- x$best_row
              df
            },
            names(step_results),
            step_results))

  out[["Stepwise History"]] <-
    stats::setNames(step_results,
             .step_name[names(step_results)])

  out[["Model Run History"]] <-
    as.data.frame(.modEnv$Store.all, stringsAsFactors = FALSE)

  return(out)
}


#' Print method for sfOperatorResult objects
#'
#' Defines a custom print method for objects of class
#' 'sfOperatorResult'.
#'
#' @param x An object of class 'sfOperatorResult'.
#' @param ... Further arguments passed to or from other methods
#'   (currently unused).
#'
#' @return Invisibly returns x.

#' @export
#'
print.sfOperatorResult <- function(x, ...) {
  cat(crayon::green$bold("\n=== Best Model Code ===\n"))
  print(x$`Final Best Code`)
  cat(crayon::green$bold("\n=== Best Model Name ===\n"))
  cat(x$`Final Best Model Name`, "\n")
  cat("\n=== Stepwise Selection History ===\n")
  print(x$`Stepwise Best Models`)
  invisible(x)
}

