#' Run an nlmixr2 model in an isolated subprocess
#'
#' Executes an nlmixr2 model fitting procedure in a separate background R
#' session using the \pkg{processx} backend. Running the model in an isolated
#' subprocess prevents the main R session from crashing and allows
#' monitoring errors, wall-time limits, and controlled output.
#'
#' @param modi Integer. A model index used to generate unique temporary filenames.
#' @param dat A data frame containing pharmacokinetic data in standard nlmixr2
#'   format for model fitting.
#' @param f An nlmixr2 model function (e.g., generated by \code{ppkmodGen(..., return.func = TRUE)}).
#' @param saem.control A \code{saemControl()} object providing estimation settings.
#' @param table.control A \code{tableControl()} object controlling table output behavior.
#' @param max_errors Integer. Maximum number of detected error messages before
#'   forcibly terminating the subprocess. Default is 100.
#' @param max_wall_time Numeric (seconds). Maximum allowed real (wall-clock)
#'   time for the subprocess before termination. Default is 86400 (24 hours).
#' @param temp_path Character. Directory where temporary files will be written.
#'   If NULL (default), the system temporary directory \code{tempdir()} is used.
#'   If a non-NULL path is supplied but does not exist, the function aborts with an error.
#' @param cleanup Logical. Whether to delete all temporary files upon completion.
#'   Default is TRUE. If FALSE, generated temporary files are preserved for
#'   debugging/troubleshooting.
#' @param verbose Logical. If TRUE, progress and diagnostic messages are
#'   printed during subprocess monitoring. Default is TRUE.
#'
#' @return A list with:
#' \describe{
#'   \item{fit.s}{The fitted nlmixr2 object, or NULL if the subprocess failed.}
#'   \item{loadError}{Logical indicating whether an error occurred (including timeout or crash).}
#' }
#'
#' @details
#' The model fitting is executed in an isolated background R process
#' (via \pkg{processx}) to prevent the main R session from crashing due to
#' instabilities in long-running nlmixr2/SAEM estimation routines or poorly
#' specified models. Output and error streams are monitored in real time, and
#' the subprocess is automatically terminated if either the error count
#' (\code{max_errors}) or the wall-time limit (\code{max_wall_time}) is
#' exceeded.
#'
#' Temporary files used to pass data and retrieve results are stored only in
#' the session-specific temporary directory (\code{tempdir()}) and are removed
#' upon completion, ensuring that no files are created in or left behind in the
#' user's working directory.
#'
#' @author Zhonghui Huang
#'
#' @examples
#' \donttest{
#' # Example: run a simple nlmixr2 model
#' pheno <- function() {
#'   ini({
#'     tcl <- log(0.008)     # typical clearance
#'     tv  <- log(0.6)       # typical volume
#'     eta.cl + eta.v ~ c(1,
#'                        0.01, 1)  # interindividual variability
#'     add.err <- 0.1        # residual variability
#'   })
#'
#'   model({
#'     cl <- exp(tcl + eta.cl)
#'     v  <- exp(tv  + eta.v)
#'     ke <- cl / v
#'     d/dt(A1) = -ke * A1
#'     cp = A1 / v
#'     cp ~ add(add.err)
#'   })
#' }
#'  run_model_in_subprocess(
#'     modi = 1,
#'     dat = pheno_sd,
#'     f = pheno,
#'     saem.control = nlmixr2est::saemControl(
#'       seed = 1234,
#'       nBurn = 100,
#'       nEm = 100,
#'       logLik = TRUE
#'     )
#'   )
#' }
#' @export

run_model_in_subprocess <- function(modi, dat, f, saem.control = NULL,
                                    table.control = NULL, max_errors = 100,
                                    max_wall_time = 86400, temp_path = NULL,
                                    cleanup = TRUE, verbose = TRUE) {
  # Required package
  if (!requireNamespace("processx", quietly = TRUE)) {
    stop(
      "The 'processx' package is required. Please install it via install.packages('processx')."
    )
  }

  # Validate and assign temporary directory
  if (is.null(temp_path)) {
    tmpdir <- tempdir()
  } else {
    if (!dir.exists(temp_path)) {
      stop("The temporary path specified in 'temp_path' does not exist.")
    }
    tmpdir <- temp_path
  }

  temp_model_data <-
    file.path(tmpdir, paste0("model_data_", modi, ".rds"))
  temp_fit_result <-
    file.path(tmpdir, paste0("fit_result_", modi, ".rds"))
  temp_script <- file.path(tmpdir, paste0("script_", modi, ".R"))

  # Save model components to RDS
  saveRDS(
    list(
      f = f,
      dat = dat,
      saem.control = saem.control,
      table.control = table.control
    ),
    file = temp_model_data
  )

  # Normalize paths for cross-platform compatibility
  temp_model_data <- normalizePath(temp_model_data, winslash = "/", mustWork = TRUE)
  temp_fit_result <- normalizePath(temp_fit_result, winslash = "/", mustWork = FALSE)

  # Create temporary R script for subprocess
  script_content <- sprintf(
    '
  suppressMessages({
    library(rxode2)
    library(nlmixr2)
  })

  model_data <- readRDS("%s")
  f <- model_data$f
  dat <- model_data$dat
  saem.control <- model_data$saem.control
  table.control <- model_data$table.control

  fit_result <- tryCatch({
    nlmixr2(
      f,
      dat,
      est = "saem",
      control = saem.control,
      table = table.control
    )
  }, error = function(e) {
    NULL
  })

  saveRDS(fit_result, "%s")
  ',
    temp_model_data,
    temp_fit_result
  )

  # Write to script file
  writeLines(script_content, temp_script)

  # Launch subprocess
  p <-
    processx::process$new("Rscript",
                          c(temp_script),
                          stdout = "|",
                          stderr = "|")

  # ---- Critical Cleanup ----
  on.exit({
    if (!is.null(p) && p$is_alive()) {
      try(p$kill(), silent = TRUE)
    }
    if (!is.null(p)) {
      try(p$wait(timeout = 1000), silent = TRUE)
    }
    if (!is.null(p)) {
      try(p$finalize(), silent = TRUE)
    }
    if (cleanup) {
    try(unlink(c(temp_model_data, temp_fit_result, temp_script), force = TRUE),
        silent = TRUE)
    }
  }, add = TRUE)
  # -----------------------------------------------------

  # -----------------------------------------------------
  # Real-time monitoring
  error_count <- 0
  loadError <- FALSE
  start_time <- Sys.time()

  while (p$is_alive()) {
    lines_out <- p$read_output_lines()
    lines_err <- p$read_error_lines()

    for (line in c(lines_out, lines_err)) {
      if (grepl("unhandled error message:", line)) {
        error_count <- error_count + 1
        if (verbose) {
          message(sprintf(
            "Model %d: Error detected %d/%d",
            modi,
            error_count,
            max_errors
          ))
        }
        if (error_count >= max_errors) {
          p$kill()
          if (verbose) {
            message(sprintf(
              "Model %d: Wall-time limit exceeded. Terminated.",
              modi
            ))
          }
          loadError <- TRUE
          break
        }
      }
    }

    if (as.numeric(difftime(Sys.time(), start_time, units = "secs")) >
        max_wall_time) {
      p$kill()
      if (verbose) {
        message(sprintf("Model %d: Wall-time limit exceeded. Terminated.", modi))
      }
      loadError <- TRUE
      break
    }

    Sys.sleep(0.05)
  }

  # Load results
  if (file.exists(temp_fit_result)) {
    fit.s <- readRDS(temp_fit_result)
  } else {
    fit.s <- NULL
    loadError <- TRUE
  }

  return(list(fit.s = fit.s, loadError = loadError))
}
