% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/lbaModel.R
\name{dlba}
\alias{dlba}
\alias{plba}
\alias{theoretical_dlba}
\alias{theoretical_plba}
\alias{lba_distributions}
\title{LBA Distribution Functions}
\usage{
dlba(rt_r, parameter_r, is_positive_drift_r, debug = FALSE)

plba(rt_r, parameter_r, is_positive_drift_r, time_parameter_r, debug = FALSE)

theoretical_dlba(
  parameter_r,
  is_positive_drift_r,
  time_parameter_r,
  debug = FALSE
)

theoretical_plba(
  parameter_r,
  is_positive_drift_r,
  time_parameter_r,
  debug = FALSE
)
}
\arguments{
\item{rt_r}{For \code{dlba} and \code{plba}: A numeric vector of
\strong{observed} response times}

\item{parameter_r}{A numeric matrix of parameters where rows represent:
\itemize{
\item Starting point variability (A)
\item Thresholds (b)
\item Mean drift rates (mean_v)
\item The standard deviation of the drift rates (sd_v)
\item The variability of the non-decision time (st0)
\item Non-decision time (t0).}
Each column represents parameters for an accumulator.}

\item{is_positive_drift_r}{A logical vector indicating whether drift rates
are strictly positive}

\item{debug}{A logical value indicating whether to print debug information.}

\item{time_parameter_r}{For theoretical functions, a numeric vector to set
minimal decision time, maximal decision time, and the step size (dt). The
internal mechanism uses this vector to set fine time points for evaluation.}
}
\value{
For all functions: A list containing:
\itemize{
\item \strong{values} - The computed distribution values
\item \strong{time_points} - The time points used (for theoretical
functions and \strong{plba})
\item Additional diagnostic information when applicable
}
}
\description{
The functions, \code{dlba}, \code{plba},
\code{theoretical_dlba}, and \code{theoretical_plba}, calculate probability
distributions for the Linear Ballistic Accumulator model:
\itemize{
\item \code{dlba} - Probability density function (PDF) for
\strong{observed} choice response times
\item \code{plba} - Cumulative distribution function (CDF) for
\strong{observed} choice response times
\item \code{theoretical_dlba} - Theoretical density function sets a
time range and computes the density for this range.
\item \code{theoretical_plba} - Theoretical cumulative distribution
function sets a time range and computes the cumulative density for this
range.
}
}
\details{
These functions provide the computational foundation for the LBA model:
\describe{
\item{\code{dlba}}{Computes the probability density for observed response
times, used during model fitting and likelihood calculations}
\item{\code{plba}}{Computes the cumulative probability for observed
response times, useful for model validation and goodness-of-fit tests}
\item{\code{theoretical_dlba}}{Computes the theoretical PDF for diagnostic
purposes and prediction}
\item{\code{theoretical_plba}}{Computes the theoretical CDF for model
validation and comparison with empirical data}
}
}
\examples{
#-------------------------------------------------#
# Example 1: theoretical_dlba and theoretical_plba
#-------------------------------------------------#
# Tiny helper to build the parameter matrix
# (rows = A, b, mean_v, sd_v, st0, t0)
param_list2mat <- function(x) do.call(rbind, x)

params <- param_list2mat(list(
  A = c(0.5, 0.5),
  b = c(1.0, 1.0),
  mean_v = c(2.0, 1.0),
  sd_v = c(1.0, 1.0),
  st0 = c(0.0, 0.0),
  t0 = c(0.2, 0.2)
))
is_pos <- rep(TRUE, ncol(params))

# Use a coarse, short grid so it runs quickly
dt <- 0.05
time_param <- c(0, 2, dt)

# Theoretical densities/cdfs for two accumulators
pdfs <- theoretical_dlba(params, is_pos, time_param)
cdfs <- theoretical_plba(params, is_pos, time_param)

# Combine to check mass and monotonicity quickly
mass_from_pdf <- sum((pdfs[[1]] + pdfs[[2]]) * dt)
tail_cdf_sum <- tail(cdfs[[1]] + cdfs[[2]], 1)

# Both should be close to 1 (within coarse-grid tolerance)
round(c(mass_from_pdf = mass_from_pdf, tail_cdf_sum = tail_cdf_sum), 3)

# Spot-check dlba/plba at a few RTs (shifted by t0)
RT <- c(0.25, 0.50, 0.75) + params[6, 1]
pl <- plba(RT, params, is_pos, time_param)
head(pl[[1]])

## ---- Extended (plots; only in interactive sessions) --------------------
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
{
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Reuse objects from above; create a denser grid for nicer plots
  dt2 <- 0.01
  time_param2 <- c(0, 5, dt2)
  DT <- seq(time_param2[1], time_param2[2], by = time_param2[3])

  pdfs2 <- theoretical_dlba(params, is_pos, time_param2)
  cdfs2 <- theoretical_plba(params, is_pos, time_param2)
  pdf_all <- (pdfs2[[1]] + pdfs2[[2]]) * dt2
  cdf_all <- cdfs2[[1]] + cdfs2[[2]]

  par(mfrow = c(2, 1), mar = c(5, 5, 2, 1))
  # PDF
  plot(DT, pdfs2[[1]] * dt2,
    type = "l",
    xlab = "DT", ylab = "Density", main = "Theoretical PDF"
  )
  lines(DT, pdfs2[[2]] * dt2, lty = 2)
  legend("topright", legend = c("Acc 1", "Acc 2"), lty = c(1, 2), bty = "n")

  # Cumulated PDF vs CDF
  plot(DT, cumsum(pdf_all),
    type = "l",
    xlab = "DT", ylab = "Cumulative", main = "Cumulated PDF and CDF",
    ylim = c(0, 1)
  )
  lines(DT, cdf_all, lty = 2)
  legend("bottomright",
    legend = c("Cumulated PDF", "CDF"),
    lty = c(1, 2), bty = "n"
  )

  # Optional: grid-comparison for plba
  RT2 <- seq(0, 3, by = 0.002) + params[6, 1]
  c1 <- plba(RT2, params, is_pos, c(0, 10, 0.01))
  c2 <- plba(RT2, params, is_pos, c(0, 5, 0.10))
  c3 <- plba(RT2, params, is_pos, c(0, 5, 0.20))

  # Okabe–Ito color-blind–safe palette
  col1 <- "#0072B2" # dt=0.01
  col2 <- "#D55E00" # dt=0.1
  col3 <- "#009E73" # dt=0.2

  par(mfrow = c(1, 1), mar = c(5, 5, 2, 2))
  plot(RT2, c1[[1]],
    type = "l", ylim = c(0, 1), xlab = "RT", ylab = "CDF",
    main = "LBA CDF Estimates under Different Time Grids", lwd = 2,
    col = col1
  )
  lines(RT2, c1[[2]], lwd = 2, lty = 2, col = col1)

  lines(RT2, c2[[1]], lwd = 2, col = col2)
  lines(RT2, c2[[2]], lwd = 2, lty = 2, col = col2)
  lines(RT2, c3[[1]], lwd = 2, col = col3)
  lines(RT2, c3[[2]], lwd = 2, lty = 2, col = col3)
  legend("bottomright",
    legend = c(
      "Acc1 (dt=0.01)", "Acc2 (dt=0.01)",
      "Acc1 (dt=0.1)", "Acc2 (dt=0.1)",
      "Acc1 (dt=0.2)", "Acc2 (dt=0.2)"
    ),
    col = c(col1, col1, col2, col2, col3, col3),
    lty = c(1, 2, 1, 2, 1, 2), lwd = 2, bty = "n"
  )
}
\dontshow{\}) # examplesIf}
}
