#' @title Function to solve for the baseline recurrent event rate given a mean frequency difference for one-sample simulations.
#' @description
#' Computes the value of the baseline recurrent event rate parameter \eqn{\lambda_0} hat yields a specified difference in mean
#' frequency function at a given event time, relative to a null value.
#' @param lambda_star Rate parameter of an exponential distribution in generating the terminal event.
#' @param null.lambda_0 Baseline recurrent event rate under the null hypothesis.
#' @param gamma_shape Shape parameter of the Gamma frailty distribution.
#' @param gamma_scale Scale parameter of the Gamma frailty distribution.
#' @param Delta Target difference in the mean frequency function at time \code{t} between the null and alternative.
#' @param t Event time (time since enrollment) at which the mean frequency difference is defined.
#'
#' @returns A numeric value giving the baseline recurrent event rate \eqn{\lambda_0} that satisfies the specified mean
#' frequency difference at time \code{t}.
#' @importFrom stats uniroot
#' @export
#'
#' @examples
#' # Toy Example: n = 800, two interim analysi planned
#' # at 50% and 75% if maximum information, along with
#' # a final analysis at 100% maximum information.
#' # Assuming the maximum information is 100,
#' # overall Type I error at 0.05 and power at 0.8.
#' find.Delta.given.power(K = 3, timing = c(0.5, 0.75, 1),
#' alpha = 0.05, power = 0.8, Imax = 100)
#' # Use the computed value from the
#' # "find.Delta.given.power()" function
#' find.lambda_0.given.Delta(lambda_star = 0.1, null.lambda_0 = 1.15,
#' gamma_shape = 2, gamma_scale = 0.5, Delta = 0.2834591, t = 2)
#'
find.lambda_0.given.Delta <- function(lambda_star, null.lambda_0, gamma_shape, gamma_scale, Delta, t){
  f = function(lambda_0){
    (lambda_0/lambda_star)*(1 - (lambda_star*t*gamma_scale + 1)^(-gamma_shape)) -
      ((null.lambda_0/lambda_star)*(1 - (lambda_star*t*gamma_scale + 1)^(-gamma_shape))-Delta)
  }
  return(uniroot(f, c(0, 5))$root)
}
