# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' @name rchol
#' @title Cholesky decomposition via R factorization.
#' @description rchol, provides the Cholesky decomposition of the symmetric and positive definite matrix \eqn{X^\top X\in\mathbb{R}^{p\times p}}, where \eqn{X\in\mathbb{R}^{n\times p}} is the input matrix.
#' @param X an \eqn{(n\times p)} matrix, with \eqn{n\geq p}. If \eqn{n< p} an error message is returned.
#' @return an upper triangular matrix of dimension \eqn{p\times p} which represents the Cholesky decomposition of \eqn{X^\top X}.
#' @examples
#'
#' set.seed(1234)
#' n <- 10
#' p <- 6
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## compute the Cholesky decomposition of X^TX
#' S <- fastQR::rchol(X = X)
#' S
#'
#' ## check
#' round(S - chol(crossprod(X)), 5)
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
NULL

#' @name qrridge_cv
#' @title Cross-validation of the RIDGE estimator for the linear regression model
#' @description qrridge_cv, or LS for linear multivariate regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|Y-XB\|_2^2,}
#' for \eqn{Y\in\mathbb{R}^{n \times q}} and \eqn{X\in\mathbb{R}^{n\times p}}, to obtain a coefficient matrix \eqn{\widehat{B}\in\mathbb{R}^{p\times q}}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param y a vector of length-\eqn{n} response vector.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param lambda a vector of lambdas.
#' @param k an integer vector defining the number of groups for CV.
#' @param seed ad integer number defining the seed for random number generation.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{coeff}{a length-\eqn{p} vector containing the solution for the parameters \eqn{\beta}.}
#' \item{fitted}{a length-\eqn{n} vector of fitted values, \eqn{\widehat{y}=X\widehat{\beta}}.}
#' \item{residuals}{a length-\eqn{n} vector of residuals, \eqn{\varepsilon=y-\widehat{y}}.}
#' \item{residuals_norm2}{the L2-norm of the residuals, \eqn{\Vert\varepsilon\Vert_2^2.}}
#' \item{y_norm2}{the L2-norm of the response variable. \eqn{\Vert y\Vert_2^2.}}
#' \item{XTX}{the matrix \eqn{X^\top X}.}
#' \item{XTy}{\eqn{X^\top y}.}
#' \item{sigma_hat}{estimated  residual variance.}
#' \item{df}{degrees of freedom.}
#' \item{Q}{\eqn{Q} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{R}{\eqn{R} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{QXTy}{\eqn{QX^\top y}, where \eqn{Q} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{\beta}}. It is only available if X_test is not NULL.}
#' }
#' @examples
#'
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- rnorm(n)
#' beta      <- rep(1, p)
#' y         <- X %*% beta + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <- fastQR::qrridge_cv(y = y, X = X, lambda = c(1,2), 
#'                                 k = 5, seed = 12, X_test = X_test, type = "QR")
#' output$coeff
#'
NULL

#' @name qrmls
#' @title Ordinary least squares for the linear multivariate regression model
#' @description qrmls, or LS for linear multivariate regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|Y-XB\|_2^2,}
#' for \eqn{Y\in\mathbb{R}^{n \times q}} and \eqn{X\in\mathbb{R}^{n\times p}}, to obtain a coefficient matrix \eqn{\widehat{B}\in\mathbb{R}^{p\times q}}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param Y a matrix of dimension \eqn{(n\times q} response variables.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{coeff}{a matrix of dimension \eqn{p\times q} containing the solution for the parameters \eqn{B}.}
#' \item{fitted}{a matrix of dimension \eqn{n\times q} of fitted values, \eqn{\widehat{Y}=X\widehat{B}}.}
#' \item{residuals}{a matrix of dimension \eqn{n\times q} of residuals, \eqn{\varepsilon=Y-\widehat{Y}}.}
#' \item{XTX}{the matrix \eqn{X^\top X}.}
#' \item{Sigma_hat}{a matrix of dimension \eqn{q\times q} containing the estimated  residual variance-covariance matrix.}
#' \item{df}{degrees of freedom.}
#' \item{R}{\eqn{R} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{XTy}{\eqn{X^\top y}.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{B}}. It is only available if X_test is not NULL.}
#' \item{PMSE}{}
#' }
#' @examples
#'
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' q         <- 3
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- matrix(rnorm(n*q), n, q)
#' B         <- matrix(0, p, q)
#' B[,1]     <- rep(1, p)
#' B[,2]     <- rep(2, p)
#' B[,3]     <- rep(-1, p)
#' Y         <- X %*% B + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <- fastQR::qrmls(Y = Y, X = X, X_test = X_test, type = "QR")
#' output$coeff
#'
NULL

#' @name qrmridge
#' @title RIDGE estimator for the linear multivariate regression model
#' @description qrmridge, or LS for linear multivariate regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|Y-XB\|_2^2,}
#' for \eqn{Y\in\mathbb{R}^{n \times q}} and \eqn{X\in\mathbb{R}^{n\times p}}, to obtain a coefficient matrix \eqn{\widehat{B}\in\mathbb{R}^{p\times q}}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param Y a matrix of dimension \eqn{(n\times q} response variables.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param lambda a vector of lambdas.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{coeff}{a matrix of dimension \eqn{p\times q} containing the solution for the parameters \eqn{B}.}
#' \item{fitted}{a matrix of dimension \eqn{n\times q} of fitted values, \eqn{\widehat{Y}=X\widehat{B}}.}
#' \item{residuals}{a matrix of dimension \eqn{n\times q} of residuals, \eqn{\varepsilon=Y-\widehat{Y}}.}
#' \item{XTX}{the matrix \eqn{X^\top X}.}
#' \item{Sigma_hat}{a matrix of dimension \eqn{q\times q} containing the estimated  residual variance-covariance matrix.}
#' \item{df}{degrees of freedom.}
#' \item{R}{\eqn{R} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{XTy}{\eqn{X^\top y}.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{B}}. It is only available if X_test is not NULL.}
#' \item{PMSE}{}
#' }
#'
#' @examples
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' q         <- 3
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- matrix(rnorm(n*q), n, q)
#' B         <- matrix(0, p, q)
#' B[,1]     <- rep(1, p)
#' B[,2]     <- rep(2, p)
#' B[,3]     <- rep(-1, p)
#' Y         <- X %*% B + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <- fastQR::qrmridge(Y = Y, X = X, lambda = 1, X_test = X_test, type = "QR")
#' output$coeff
#'
NULL

#' @name qrmridge_cv
#' @title Cross-validation of the RIDGE estimator for the linear multivariate regression model
#' @description qrmridge_cv, or LS for linear multivariate regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|Y-XB\|_2^2,}
#' for \eqn{Y\in\mathbb{R}^{n \times q}} and \eqn{X\in\mathbb{R}^{n\times p}}, to obtain a coefficient matrix \eqn{\widehat{B}\in\mathbb{R}^{p\times q}}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param Y a matrix of dimension \eqn{(n\times q} response variables.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param lambda a vector of lambdas.
#' @param k an integer vector defining the number of groups for CV.
#' @param seed ad integer number defining the seed for random number generation.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{coeff}{a matrix of dimension \eqn{p\times q} containing the solution for the parameters \eqn{B}.}
#' \item{fitted}{a matrix of dimension \eqn{n\times q} of fitted values, \eqn{\widehat{Y}=X\widehat{B}}.}
#' \item{residuals}{a matrix of dimension \eqn{n\times q} of residuals, \eqn{\varepsilon=Y-\widehat{Y}}.}
#' \item{XTX}{the matrix \eqn{X^\top X}.}
#' \item{Sigma_hat}{a matrix of dimension \eqn{q\times q} containing the estimated  residual variance-covariance matrix.}
#' \item{df}{degrees of freedom.}
#' \item{R}{\eqn{R} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{XTy}{\eqn{X^\top y}.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{B}}. It is only available if X_test is not NULL.}
#' \item{PMSE}{}
#' }
#'
#' @examples
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' q         <- 3
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- matrix(rnorm(n*q), n, q)
#' B         <- matrix(0, p, q)
#' B[,1]     <- rep(1, p)
#' B[,2]     <- rep(2, p)
#' B[,3]     <- rep(-1, p)
#' Y         <- X %*% B + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <- fastQR::qrmridge_cv(Y = Y, X = X, lambda = c(1,2),
#'                                  k = 5, seed = 12, X_test = X_test, type = "QR")
#' output$coeff
#' 
NULL

#' @name qrls
#' @title Ordinary least squares for the linear regression model
#' @description qrls, or LS for linear regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|y-X\beta\|_2^2,}
#' for \eqn{y\in\mathbb{R}^n} and \eqn{X\in\mathbb{R}^{n\times p}}, to obtain a coefficient vector \eqn{\widehat{\beta}\in\mathbb{R}^p}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param y a vector of length-\eqn{n} response vector.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{coeff}{a length-\eqn{p} vector containing the solution for the parameters \eqn{\beta}.}
#' \item{fitted}{a length-\eqn{n} vector of fitted values, \eqn{\widehat{y}=X\widehat{\beta}}.}
#' \item{residuals}{a length-\eqn{n} vector of residuals, \eqn{\varepsilon=y-\widehat{y}}.}
#' \item{residuals_norm2}{the L2-norm of the residuals, \eqn{\Vert\varepsilon\Vert_2^2.}}
#' \item{y_norm2}{the L2-norm of the response variable. \eqn{\Vert y\Vert_2^2.}}
#' \item{XTX_Qmat}{\eqn{Q} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{XTX_Rmat}{\eqn{R} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{QXTy}{\eqn{QX^\top y}, where \eqn{Q} matrix of the QR decomposition of the matrix \eqn{X^\top X}.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{\beta}}. It is only available if X_test is not NULL.}
#' }
#' @examples
#'
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- rnorm(n, sd = 0.5)
#' beta      <- rep(0, p)
#' beta[1:3] <- 1
#' beta[4:5] <- 2
#' y         <- X %*% beta + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <-  fastQR::qrls(y = y, X = X, X_test = X_test)
#' output$coeff
#'
qrls <- function(y, X, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrls`, y, X, X_test, type)
}

#' @name qrridge
#' @title RIDGE estimation for the linear regression model
#' @description lmridge, or RIDGE for linear regression models, solves the following penalized optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{n}\|y-X\beta\|_2^2+\lambda\Vert\beta\Vert_2^2,}
#' to obtain a coefficient vector \eqn{\widehat{\beta}\in\mathbb{R}^{p}}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param y a vector of length-\eqn{n} response vector.
#' @param X an \eqn{(n\times p)} matrix of predictors.
#' @param lambda a vector of lambdas.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @return A named list containing \describe{
#' \item{mean_y}{mean of the response variable.}
#' \item{mean_X}{a length-\eqn{p} vector containing the mean of each column of the design matrix.}
#' \item{path}{the whole path of estimated regression coefficients.}
#' \item{ess}{explained sum of squares for the whole path of estimated coefficients.}
#' \item{GCV}{generalized cross-validation for the whole path of lambdas.}
#' \item{GCV_min}{minimum value of GCV.}
#' \item{GCV_idx}{inded corresponding to the minimum values of GCV.}
#' \item{coeff}{a length-\eqn{p} vector containing the solution for the parameters \eqn{\beta} which corresponds to the minimum of GCV.}
#' \item{lambda}{the vector of lambdas.}
#' \item{scales}{the vector of standard deviations of each column of the design matrix.}
#' }
#' @examples
#'
#' ## generate sample data
#' set.seed(10)
#' n         <- 30
#' p         <- 6
#' X         <- matrix(rnorm(n * p, 1), n, p)
#' X[,1]     <- 1
#' eps       <- rnorm(n, sd = 0.5)
#' beta      <- rep(0, p)
#' beta[1:3] <- 1
#' beta[4:5] <- 2
#' y         <- X %*% beta + eps
#' X_test    <- matrix(rnorm(5 * p, 1), 5, p)
#' output    <-  fastQR::qrridge(y = y, X = X,
#'                               lambda = 0.2,
#'                               X_test = X_test)
#' output$coeff
#'
qrridge <- function(y, X, lambda, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrridge`, y, X, lambda, X_test, type)
}

#' @name qrchol
#' @title Cholesky decomposition via QR factorization.
#' @description qrchol, provides the Cholesky decomposition of the symmetric and positive definite matrix \eqn{X^\top X\in\mathbb{R}^{p\times p}}, where \eqn{X\in\mathbb{R}^{n\times p}} is the input matrix.
#' @param X an \eqn{(n\times p)} matrix.
#' @param nb number of blocks for the recursive block QR decomposition, default is NULL.
#' @return an upper triangular matrix of dimension \eqn{p\times p} which represents the Cholesky decomposition of \eqn{X^\top X}.
qrchol <- function(X, nb = NULL) {
    .Call(`_fastQR_qrchol`, X, nb)
}

#' @name qrsolve
#' @title Solution of linear system of equations, via the QR decomposition.
#' @description solves systems of equations \eqn{Ax=b}, for \eqn{A\in\mathbb{R}^{n\times p}} and \eqn{b\in\mathbb{R}^n}, via the QR decomposition.
#' @param A an \eqn{(n\times p)} full column rank matrix.
#' @param b a vector of dimension \eqn{n}.
#' @param type either "QR" or "R". Specifies the type of decomposition to use: "QR" for the QR decomposition or "R" for the Cholesky factorization of \eqn{A^\top A}. The default is "QR".
#' @param nb number of blocks for the recursive block QR decomposition, default is NULL.
#' @return x a vector of dimension \eqn{p} that satisfies \eqn{Ax=b}.
#' @examples
#'
#' ## generate sample data
#' set.seed(1234)
#' n <- 10
#' p <- 4
#' A <- matrix(rnorm(n * p, 1), n, p)
#' b <- rnorm(n)
#'
#' ## solve the system of linear equations using qr
#' x1 <- fastQR::qrsolve(A = A, b = b)
#' x1
#'
#' ## solve the system of linear equations using rb qr
#' x2 <- fastQR::qrsolve(A = A, b = b, nb = 2)
#' x2
#'
#' ## check
#' round(x1 - solve(crossprod(A)) %*% crossprod(A, b), 5)
#' round(x2 - solve(crossprod(A)) %*% crossprod(A, b), 5)
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qrsolve <- function(A, b, type = NULL, nb = NULL) {
    .Call(`_fastQR_qrsolve`, A, b, type, nb)
}

rchol <- function(X) {
    .Call(`_fastQR_rchol`, X)
}

qrridge_cv <- function(y, X, lambda, k = NULL, seed = NULL, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrridge_cv`, y, X, lambda, k, seed, X_test, type)
}

qrmls <- function(Y, X, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrmls`, Y, X, X_test, type)
}

qrmridge <- function(Y, X, lambda, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrmridge`, Y, X, lambda, X_test, type)
}

qrmridge_cv <- function(Y, X, lambda, k = NULL, seed = NULL, X_test = NULL, type = NULL) {
    .Call(`_fastQR_qrmridge_cv`, Y, X, lambda, k, seed, X_test, type)
}

#' Multiply Q by a vector using a QR decomposition
#'
#' Computes \eqn{Q^\top y}, where \eqn{Q} is the orthogonal matrix from the
#' QR decomposition stored in compact (Householder) form.
#'
#' @param qr numeric matrix containing the QR decomposition in compact form
#'   (as returned by \code{qr_fast()}).
#' @param tau numeric vector of Householder coefficients.
#' @param y numeric vector of length \eqn{n}.
#'
#' @return a numeric vector equal to \eqn{Q^\top y}.
#'
#' @details
#' The orthogonal matrix \eqn{Q} is not formed explicitly. The product
#' \eqn{Q^\top y} is computed efficiently using the Householder reflectors
#' stored in \code{qr} and \code{tau}.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' qr_res <- fastQR::qr_fast(X)
#' res1   <- fastQR::qr_Qty(qr = qr_res$qr, tau = qr_res$qraux, y = y)
#'
#' ## reference computation
#' Q    <- base::qr.Q(base::qr(X), complete = TRUE)
#' res2 <- crossprod(Q, y)
#'
#' max(abs(res1 - drop(res2)))
#'
qr_Qty <- function(qr, tau, y) {
    .Call(`_fastQR_qr_Qty`, qr, tau, y)
}

#' Multiply Q by a vector using a QR decomposition
#'
#' Computes \eqn{Q y}, where \eqn{Q} is the orthogonal matrix from the
#' QR decomposition stored in compact (Householder) form.
#'
#' @param qr numeric matrix containing the QR decomposition in compact form
#'   (as returned by \code{qr_fast()}).
#' @param tau numeric vector of Householder coefficients.
#' @param y numeric vector of length \eqn{n}.
#'
#' @return a numeric vector equal to \eqn{Q y}.
#'
#' @details
#' The orthogonal matrix \eqn{Q} is not formed explicitly. The product
#' \eqn{Q y} is computed efficiently using the Householder reflectors
#' stored in \code{qr} and \code{tau}.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' qr_res <- fastQR::qr_fast(X)
#' res1   <- fastQR::qr_Qy(qr = qr_res$qr, tau = qr_res$qraux, y = y)
#'
#' ## reference computation
#' Q    <- base::qr.Q(base::qr(X), complete = TRUE)
#' res2 <- Q %*% y
#'
#' max(abs(res1 - drop(res2)))
#'
qr_Qy <- function(qr, tau, y) {
    .Call(`_fastQR_qr_Qy`, qr, tau, y)
}

#' Compute least-squares coefficients from a QR decomposition
#'
#' Computes the coefficient vector \eqn{\widehat\beta} solving the
#' least-squares problem \eqn{\min_\beta \|y - X\beta\|_2},
#' using a QR decomposition stored in compact (Householder) form.
#'
#' @param qr numeric matrix containing the QR decomposition of \eqn{X}
#'   in compact form (as returned by \code{qr_fast()}).
#' @param tau numeric vector of Householder coefficients.
#' @param y numeric response vector of length \eqn{n}.
#' @param pivot optional integer vector of length \eqn{p} containing the
#'   1-based column permutation used during the QR factorization. If supplied,
#'   the returned coefficients are reordered to match the original column order.
#' @param rank optional integer specifying the numerical rank of \eqn{X}. If
#'   supplied, only the leading \code{rank} components are used in the
#'   triangular solve.
#'
#' @return a numeric vector of regression coefficients.
#'
#' @details
#' The coefficients are obtained by first computing \eqn{Q^\top y}
#' and then solving the resulting upper-triangular system involving
#' the matrix \eqn{R}. The orthogonal matrix \eqn{Q} is never formed
#' explicitly.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' qr_res <- fastQR::qr_fast(X)
#' coef1  <- fastQR::qr_coef(qr = qr_res$qr, tau = qr_res$qraux, y = y)
#'
#' ## reference computation
#' coef2 <- base::qr.coef(base::qr(X), y)
#'
#' max(abs(coef1 - coef2))
#'
qr_coef <- function(qr, tau, y, pivot = NULL, rank = NULL) {
    .Call(`_fastQR_qr_coef`, qr, tau, y, pivot, rank)
}

#' Compute fitted values from a QR decomposition
#'
#' Computes the fitted values \eqn{\widehat y = X\widehat\beta} for a linear
#' least-squares problem using a QR decomposition stored in compact
#' (Householder) form.
#'
#' @param qr Numeric matrix containing the QR decomposition of \eqn{X}
#'   in compact form (as returned by \code{qr_fast()}).
#' @param tau numeric vector of Householder coefficients.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector of fitted values \eqn{\hat y}.
#'
#' @details
#' The fitted values are computed as
#' \deqn{\widehat y = Q Q^\top y}
#' without explicitly forming the orthogonal matrix \eqn{Q}. The
#' computation relies on the Householder reflectors stored in
#' \code{qr} and \code{tau}.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' qr_res <- fastQR::qr_fast(X)
#' yhat1  <- fastQR::qr_fitted(qr = qr_res$qr, tau = qr_res$qraux, y = y)
#'
#' ## reference computation
#' yhat2 <- base::qr.fitted(base::qr(X), y)
#'
#' max(abs(yhat1 - yhat2))
#'
qr_fitted <- function(qr, tau, y) {
    .Call(`_fastQR_qr_fitted`, qr, tau, y)
}

#' Compute residuals from a QR decomposition
#'
#' Computes the residual vector \eqn{r = y - \widehat y} for a linear
#' least-squares problem using a QR decomposition stored in compact
#' (Householder) form.
#'
#' @param qr numeric matrix containing the QR decomposition of \eqn{X}
#'   in compact form (as returned by \code{qr_fast()}).
#' @param tau numeric vector of Householder coefficients.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector of residuals of dimension \eqn{n}.
#'
#' @details
#' The residuals are computed as
#' \deqn{r = y - Q Q^\top y}
#' without explicitly forming the orthogonal matrix \eqn{Q}. The
#' computation relies on the Householder reflectors stored in
#' \code{qr} and \code{tau}.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' qr_res <- fastQR::qr_fast(X)
#' r1     <- fastQR::qr_resid(qr = qr_res$qr, tau = qr_res$qraux, y = y)
#'
#' ## reference computation
#' r2 <- base::qr.resid(base::qr(X), y)
#'
#' max(abs(r1 - r2))
#'
qr_resid <- function(qr, tau, y) {
    .Call(`_fastQR_qr_resid`, qr, tau, y)
}

#' Compute least-squares coefficients using QR decomposition
#'
#' Computes the coefficient vector \eqn{\hat\beta} solving the
#' least-squares problem \eqn{\min_\beta \|y - X\beta\|_2},
#' using a QR decomposition computed internally.
#'
#' @param X numeric matrix of dimension \eqn{n \times p}.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector of regression coefficients.
#'
#' @details
#' The QR decomposition of \eqn{X} is computed internally. The coefficients
#' are obtained by first computing \eqn{Q^\top y} and then solving the
#' resulting upper-triangular system involving the matrix \eqn{R}.
#' The orthogonal matrix \eqn{Q} is never formed explicitly.
#'
#' This function is intended as a convenience wrapper for least-squares
#' estimation when the explicit QR factors are not required.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' coef1 <- fastQR::qr_lse_coef(X, y)
#'
#' ## reference computation
#' coef2 <- base::qr.coef(base::qr(X), y)
#'
#' max(abs(coef1 - coef2))
#'
qr_lse_coef <- function(X, y) {
    .Call(`_fastQR_qr_lse_coef`, X, y)
}

#' Compute fitted values using QR decomposition
#'
#' Computes the fitted values \eqn{\hat y = X\hat\beta} for a linear
#' least-squares problem using a QR decomposition computed internally.
#'
#' @param X numeric matrix of dimension \eqn{n \times p}.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector of fitted values \eqn{\hat y}.
#'
#' @details
#' The QR decomposition of \eqn{X} is computed internally. The fitted
#' values are obtained as
#' \deqn{\widehat y = Q Q^\top y}
#' without explicitly forming the orthogonal matrix \eqn{Q}.
#'
#' This function is intended as a convenience wrapper for least-squares
#' computations when the explicit QR factors are not required.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' yhat1 <- fastQR::qr_lse_fitted(X, y)
#'
#' ## reference computation
#' yhat2 <- base::qr.fitted(base::qr(X), y)
#'
#' max(abs(yhat1 - yhat2))
#'
qr_lse_fitted <- function(X, y) {
    .Call(`_fastQR_qr_lse_fitted`, X, y)
}

#' Compute residuals using QR decomposition
#'
#' Computes the residual vector \eqn{r = y - \hat y} for a linear
#' least-squares problem using a QR decomposition computed internally.
#'
#' @param X numeric matrix of dimension \eqn{n \times p}.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector of residuals.
#'
#' @details
#' The QR decomposition of \eqn{X} is computed internally. The residuals
#' are obtained as
#' \deqn{r = y - Q Q^\top y}
#' without explicitly forming the orthogonal matrix \eqn{Q}.
#'
#' This function is intended as a convenience wrapper for least-squares
#' computations when the explicit QR factors are not required.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' r1 <- fastQR::qr_lse_resid(X, y)
#'
#' ## reference computation
#' r2 <- base::qr.resid(base::qr(X), y)
#'
#' max(abs(r1 - r2))
#'
qr_lse_resid <- function(X, y) {
    .Call(`_fastQR_qr_lse_resid`, X, y)
}

#' Compute Q'y for a least-squares problem
#'
#' Computes the product \eqn{Q^\top y}, where \eqn{Q} is the orthogonal
#' matrix from the QR decomposition of the design matrix \eqn{X}.
#'
#' @param X numeric matrix of dimension \eqn{n \times p}.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector equal to \eqn{Q^\top y}.
#'
#' @details
#' The QR decomposition of \eqn{X} is computed internally, and the
#' orthogonal matrix \eqn{Q} is never formed explicitly. The product
#' \eqn{Q^\top y} is evaluated efficiently using Householder reflectors.
#'
#' This function is intended as a convenience wrapper for least-squares
#' computations when the explicit QR factors are not required.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' res1 <- fastQR::qr_lse_Qty(X, y)
#'
#' ## reference computation
#' res2 <- crossprod(base::qr.Q(base::qr(X), complete = TRUE), y)
#'
#' max(abs(res1 - drop(res2)))
#'
qr_lse_Qty <- function(X, y) {
    .Call(`_fastQR_qr_lse_Qty`, X, y)
}

#' Compute Qy for a least-squares problem
#'
#' Computes the product \eqn{Q y}, where \eqn{Q} is the orthogonal
#' matrix from the QR decomposition of the design matrix \eqn{X}.
#'
#' @param X numeric matrix of dimension \eqn{n \times p}.
#' @param y numeric response vector of length \eqn{n}.
#'
#' @return a numeric vector equal to \eqn{Q y}.
#'
#' @details
#' The QR decomposition of \eqn{X} is computed internally, and the
#' orthogonal matrix \eqn{Q} is never formed explicitly. The product
#' \eqn{Q y} is evaluated efficiently using Householder reflectors.
#'
#' This function is intended as a convenience wrapper for least-squares
#' computations when the explicit QR factors are not required.
#'
#' @examples
#' set.seed(1)
#' n <- 10; p <- 4
#' X <- matrix(rnorm(n * p), n, p)
#' y <- rnorm(n)
#'
#' res1 <- fastQR::qr_lse_Qy(X, y)
#'
#' ## reference computation
#' res2 <- base::qr.Q(base::qr(X), complete = TRUE) %*% y
#'
#' max(abs(res1 - drop(res2)))
#'
qr_lse_Qy <- function(X, y) {
    .Call(`_fastQR_qr_lse_Qy`, X, y)
}

#' @name qr
#' @title The QR factorization of a matrix
#' @description qr provides the QR factorization of the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The QR factorization of the matrix \eqn{X} returns the matrices \eqn{Q\in\mathbb{R}^{n\times n}} and \eqn{R\in\mathbb{R}^{n\times p}} such that \eqn{X=QR}. See Golub and Van Loan (2013) for further details on the method.
#' @param X a \eqn{n\times p} matrix.
#' @param type either "givens" or "householder".
#' @param nb integer. Defines the number of block in the block recursive QR decomposition. See Golud and van Loan (2013).
#' @param complete logical expression of length 1. Indicates whether an arbitrary orthogonal completion of the \eqn{Q} matrix is to be made, or whether the \eqn{R} matrix is to be completed by binding zero-value rows beneath the square upper triangle.
#' @return A named list containing \describe{
#' \item{Q}{the Q matrix.}
#' \item{R}{the R matrix.}
#' }
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 10
#' p <- 6
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## QR factorization via Givens rotation
#' output <- qr(X, type = "givens", complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## check
#' round(Q %*% R - X, 5)
#' max(abs(Q %*% R - X))
#'
#' ## QR factorization via Householder rotation
#' output <- qr(X, type = "householder", complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## check
#' round(Q %*% R - X, 5)
#' max(abs(Q %*% R - X))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
NULL

qr <- function(X, complete = NULL, type = NULL, nb = NULL) {
    .Call(`_fastQR_qr`, X, complete, type, nb)
}

#' @name qrupdate
#' @title Fast updating of the QR factorization
#' @description qrupdate provides the update of the QR factorization after the addition of \eqn{m>1} rows or columns to the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The QR factorization of the matrix \eqn{X} returns the matrices \eqn{Q\in\mathbb{R}^{n\times n}} and \eqn{R\in\mathbb{R}^{n\times p}} such that \eqn{X=QR}. The \eqn{Q} and \eqn{R} matrices are factorized as \eqn{Q=\begin{bmatrix}Q_1&Q_2\end{bmatrix}} and \eqn{R=\begin{bmatrix}R_1\\R_2\end{bmatrix}}, with \eqn{Q_1\in\mathbb{R}^{n\times p}}, \eqn{Q_2\in\mathbb{R}^{n\times (n-p)}} such that \eqn{Q_1^{\top}Q_2=Q_2^\top Q_1=0} and \eqn{R_1\in\mathbb{R}^{p\times p}} upper triangular matrix and \eqn{R_2\in\mathbb{R}^{(n-p)\times p}}. qrupdate accepts in input the matrices \eqn{Q} and either the complete matrix \eqn{R} or the reduced one, \eqn{R_1}. See Golub and Van Loan (2013) for further details on the method.
#' @param Q a \eqn{n\times p} matrix.
#' @param R a \eqn{p\times p} upper triangular matrix.
#' @param k position where the columns or the rows are added.
#' @param U either a \eqn{n\times m} matrix or a \eqn{p\times m} matrix of columns or rows to be added.
#' @param type either 'row' of 'column', for adding rows or columns. Default is 'column'.
#' @param fast fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.
#' @param complete logical expression of length 1. Indicates whether an arbitrary orthogonal completion of the \eqn{Q} matrix is to be made, or whether the \eqn{R} matrix is to be completed by binding zero-value rows beneath the square upper triangle.
#' @return A named list containing \describe{
#' \item{Q}{the updated Q matrix.}
#' \item{R}{the updated R matrix.}
#' }
#'
#' @examples
#' ## Add one column
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## create column u to be added
#' k  <- p+1
#' u  <- matrix(rnorm(n), n, 1)
#' X1 <- cbind(X, u)
#'
#' ## update the QR decomposition
#'out <- fastQR::qrupdate(Q = Q, R = R,
#'                        k = k, U = u,
#'                        type = "column",
#'                        fast = FALSE,
#'                        complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Add m columns
#' ## create data: n > p
#' set.seed(1234)
#' n <- 10
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## create the matrix of two columns to be added
#' ## in position 2
#' k  <- 2
#' m  <- 2
#' U  <- matrix(rnorm(n*m), n, m)
#' X1 <- cbind(X[,1:(k-1)], U, X[,k:p])
#'
#' # update the QR decomposition
#' out <- fastQR::qrupdate(Q = Q, R = R,
#'                         k = k, U = U, type = "column",
#'                        fast = FALSE, complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Add one row
#' ## create data: n > p
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create the row u to be added
#' u  <- matrix(data = rnorm(p), p, 1)
#' k  <- n+1
#' if (k<=n) {
#'   X1 <- rbind(rbind(X[1:(k-1), ], t(u)), X[k:n, ])
#' } else {
#'   X1 <- rbind(rbind(X, t(u)))
#' }
#'
#' ## update the QR decomposition
#' out <- fastQR::qrupdate(Q = Q, R = R,
#'                         k = k, U = u,
#'                         type = "row",
#'                         complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Add m rows
#' ## create data: n > p
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create the matrix of rows U to be added:
#' ## two rows in position 5
#' m  <- 2
#' U  <- matrix(data = rnorm(p*m), p, m)
#' k  <- 5
#' if (k<=n) {
#'   X1 <- rbind(rbind(X[1:(k-1), ], t(U)), X[k:n, ])
#' } else {
#'   X1 <- rbind(rbind(X, t(U)))
#' }
#'
#' ## update the QR decomposition
#' out <- fastQR::qrupdate(Q = Q, R = R,
#'                         k = k, U = U,
#'                         type = "row",
#'                         complete = FALSE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qrupdate <- function(Q, R, k, U, type = NULL, fast = NULL, complete = NULL) {
    .Call(`_fastQR_qrupdate`, Q, R, k, U, type, fast, complete)
}

#' @name qrdowndate
#' @title Fast downdating of the QR factorization
#' @description qrdowndate provides the update of the QR factorization after the deletion of \eqn{m>1} rows or columns to the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The QR factorization of the matrix \eqn{X\in\mathbb{R}^{n\times p}} returns the matrices \eqn{Q\in\mathbb{R}^{n\times n}} and \eqn{R\in\mathbb{R}^{n\times p}} such that \eqn{X=QR}. The \eqn{Q} and \eqn{R} matrices are factorized as \eqn{Q=\begin{bmatrix}Q_1&Q_2\end{bmatrix}} and \eqn{R=\begin{bmatrix}R_1\\R_2\end{bmatrix}}, with \eqn{Q_1\in\mathbb{R}^{n\times p}}, \eqn{Q_2\in\mathbb{R}^{n\times (n-p)}} such that \eqn{Q_1^{\top}Q_2=Q_2^\top Q_1=0} and \eqn{R_1\in\mathbb{R}^{p\times p}} upper triangular matrix and \eqn{R_2\in\mathbb{R}^{(n-p)\times p}}. qrupdate accepts in input the matrices \eqn{Q} and either the complete matrix \eqn{R} or the reduced one, \eqn{R_1}. See Golub and Van Loan (2013) for further details on the method.
#' @param Q a \eqn{n\times n} matrix.
#' @param R a \eqn{n\times p} upper triangular matrix.
#' @param k position where the columns or the rows are removed.
#' @param m number of columns or rows to be removed. Default is \eqn{m=1}.
#' @param type either 'row' of 'column', for deleting rows or columns. Default is 'column'.
#' @param fast fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.
#' @param complete logical expression of length 1. Indicates whether an arbitrary orthogonal completion of the \eqn{Q} matrix is to be made, or whether the \eqn{R} matrix is to be completed by binding zero-value rows beneath the square upper triangle.
#' @return A named list containing \describe{
#' \item{Q}{the updated Q matrix.}
#' \item{R}{the updated R matrix.}
#' }
#'
#' @examples
#' ## Remove one column
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## select the column to be deleted
#' ## from X and update X
#' k  <- 2
#' X1 <- X[, -k]
#'
#' ## downdate the QR decomposition
#' out <- fastQR::qrdowndate(Q = Q, R = R,
#'                           k = k, m = 1,
#'                           type = "column",
#'                           fast = FALSE,
#'                           complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Remove m columns
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## select the column to be deleted from X
#' ## and update X
#' m  <- 2
#' k  <- 2
#' X1 <- X[, -c(k,k+m-1)]
#'
#' ## downdate the QR decomposition
#' out <- fastQR::qrdowndate(Q = Q, R = R,
#'                           k = k, m = 2,
#'                           type = "column",
#'                           fast = TRUE,
#'                           complete = FALSE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Remove one row
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## select the row to be deleted from X and update X
#' k  <- 5
#' X1 <- X[-k,]
#'
#' ## downdate the QR decomposition
#' out <- fastQR::qrdowndate(Q = Q, R = R,
#'                           k = k, m = 1,
#'                           type = "row",
#'                           fast = FALSE,
#'                           complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' ## Remove m rows
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## select the rows to be deleted from X and update X
#' k  <- 5
#' m  <- 2
#' X1 <- X[-c(k,k+1),]
#'
#' ## downdate the QR decomposition
#' out <- fastQR::qrdowndate(Q = Q, R = R,
#'                           k = k, m = m,
#'                           type = "row",
#'                           fast = FALSE,
#'                           complete = TRUE)
#'
#' ## check
#' round(out$Q %*% out$R - X1, 5)
#' max(abs(out$Q %*% out$R - X1))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qrdowndate <- function(Q, R, k, m = NULL, type = NULL, fast = NULL, complete = NULL) {
    .Call(`_fastQR_qrdowndate`, Q, R, k, m, type, fast, complete)
}

qraddcol <- function(Q, R, k, u) {
    .Call(`_fastQR_qraddcol`, Q, R, k, u)
}

qraddmcols <- function(Q, R, k, U) {
    .Call(`_fastQR_qraddmcols`, Q, R, k, U)
}

qraddrow <- function(Q, R, k, u) {
    .Call(`_fastQR_qraddrow`, Q, R, k, u)
}

qraddmrows <- function(Q, R, k, U) {
    .Call(`_fastQR_qraddmrows`, Q, R, k, U)
}

qrdeleterow <- function(Q, R, k) {
    .Call(`_fastQR_qrdeleterow`, Q, R, k)
}

qrdeletemrows <- function(Q, R, k, m) {
    .Call(`_fastQR_qrdeletemrows`, Q, R, k, m)
}

qrdeletecol <- function(Q, R, k) {
    .Call(`_fastQR_qrdeletecol`, Q, R, k)
}

qrdeletemcols_adj <- function(Q, R, k, m) {
    .Call(`_fastQR_qrdeletemcols_adj`, Q, R, k, m)
}

#' @name qr_thin
#' @title Fast thin QR decomposition
#' @description qr_thin provides the thin QR factorization of the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The thin QR factorization of the matrix \eqn{X} returns the matrices \eqn{Q\in\mathbb{R}^{n\times p}} and the upper triangular matrix \eqn{R\in\mathbb{R}^{p\times p}} such that \eqn{X=QR}. See Golub and Van Loan (2013) for further details on the method.
#' @param X a \eqn{n\times p} matrix with \eqn{n>p}.
#' @return A named list containing \describe{
#' \item{Q}{the Q matrix.}
#' \item{R}{the R matrix.}
#' }
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the thin QR factorization
#' output <- qr_thin(X = X)
#' Q      <- output$Q
#' R      <- output$R
#'
#' ## check
#' max(abs(Q %*% R - X))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_thin <- function(X) {
    .Call(`_fastQR_qr_thin`, X)
}

#' @name qr_fast
#' @title Fast full QR decomposition
#' @description qr_fast provides the fast QR factorization of the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The full QR factorization of the matrix \eqn{X} returns the matrices \eqn{Q\in\mathbb{R}^{n\times p}} and the upper triangular matrix \eqn{R\in\mathbb{R}^{p\times p}} such that \eqn{X=QR}. See Golub and Van Loan (2013) for further details on the method.
#' @param X a \eqn{n\times p} matrix with \eqn{n>p}.
#' @param tol the tolerance for detecting linear dependencies in the columns of \eqn{X}.
#' @param pivot a logical value indicating whether to pivot the columns of \eqn{X}. Defaults to FALSE, meaning no pivoting is performed.
#' @return A named list containing \describe{
#' \item{qr}{a matrix with the same dimensions as \eqn{X}. The upper triangle contains the \eqn{R} of the decomposition and the lower triangle contains information on the \eqn{Q} of the decomposition (stored in compact form).}
#' \item{qraux}{a vector of length ncol(x) which contains additional information on \eqn{Q}.}
#' \item{rank}{the rank of \eqn{X} as computed by the decomposition.}
#' \item{pivot}{information on the pivoting strategy used during the decomposition.}
#' \item{pivoted}{a boolean variable returning one if the pivoting has been performed and zero otherwise.}
#' }
#' @details The QR decomposition plays an important role in many statistical techniques. In particular it can be used to solve the equation \eqn{Ax=b} for given matrix \eqn{A\in\mathbb{R}^{n\times p}} and vectors \eqn{x\in\mathbb{R}^{p}} and \eqn{b\in\mathbb{R}^{n}}. It is useful for computing regression coefficients and in applying the Newton-Raphson algorithm.
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## reconstruct the reduced Q and R matrices
#' ## reduced Q matrix
#' Q1 <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux,
#'            rank = qr_res$rank, complete = FALSE)
#' Q1
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q1)-diag(1, p)))
#'
#' ## complete Q matrix
#' Q2 <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux,
#'            rank = NULL, complete = TRUE)
#' Q2
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q2)-diag(1, n)))
#'
#' ## reduced R matrix
#' R1 <- qr_R(qr = qr_res$qr,
#'            rank = NULL,
#'            complete = FALSE)
#'
#' ## check that X^TX = R^TR
#' ## get the permutation matrix
#' P <- qr_pivot2perm(pivot = qr_res$pivot)
#' max(abs(crossprod(R1 %*% P) - crossprod(X)))
#' max(abs(crossprod(R1) - crossprod(X %*% t(P))))
#'
#' ## complete R matrix
#' R2 <- qr_R(qr = qr_res$qr,
#'            rank = NULL,
#'            complete = TRUE)
#'
#' ## check that X^TX = R^TR
#' ## get the permutation matrix
#' P <- qr_pivot2perm(pivot = qr_res$pivot)
#' max(abs(crossprod(R2 %*% P) - crossprod(X)))
#' max(abs(crossprod(R2) - crossprod(X %*% t(P))))
#'
#' ## check that X = Q %*% R
#' max(abs(Q2 %*% R2 %*% P - X))
#' max(abs(Q1 %*% R1 %*% P - X))
#'
#' ## create data: n > p
#' set.seed(1234)
#' n <- 120
#' p <- 75
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X, pivot = FALSE)
#'
#' ## reconstruct the reduced Q and R matrices
#' ## reduced Q matrix
#' Q1 <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux,
#'            rank = p,
#'            complete = FALSE)
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q1)-diag(1, p)))
#'
#' ## complete Q matrix
#' Q2 <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux,
#'            rank = NULL, complete = TRUE)
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q2)-diag(1, n)))
#'
#' ## reduced R matrix
#' R1 <- qr_R(qr = qr_res$qr,
#'            rank = NULL,
#'            complete = FALSE)
#'
#'
#' ## check that X^TX = R^TR
#' max(abs(crossprod(R1) - crossprod(X)))
#'
#' ## complete R matrix
#' R2 <- qr_R(qr = qr_res$qr,
#'            rank = NULL,
#'            complete = TRUE)
#'
#' ## check that X^TX = R^TR
#' max(abs(crossprod(R2) - crossprod(X)))
#'
#' ## check that X^TX = R^TR
#' max(abs(crossprod(R2) - crossprod(X)))
#' max(abs(crossprod(R2) - crossprod(X)))
#' max(abs(crossprod(R1) - crossprod(X)))
#'
#' # check that X = Q %*% R
#' max(abs(Q2 %*% R2 - X))
#' max(abs(Q1 %*% R1 - X))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_fast <- function(X, tol = NULL, pivot = NULL) {
    .Call(`_fastQR_qr_fast`, X, tol, pivot)
}

#' @name qr_Q
#' @title Reconstruct the Q, matrix from a QR object.
#' @description returns the \eqn{Q} matrix of the full QR decomposition. If \eqn{r = \mathrm{rank}(X) < p}, then only the reduced \eqn{Q \in \mathbb{R}^{n \times r}} matrix is returned.
#' @param qr object representing a QR decomposition. This will typically have come from a previous call to qr.
#' @param tau a vector of length \eqn{ncol(X)} which contains additional information on \eqn{Q}. It corresponds to qraux from a previous call to qr.
#' @param rank the rank of x as computed by the decomposition.
#' @param complete logical flag (length 1). Indicates whether to compute the full \eqn{Q \in \bold{R}^{n \times n}} or the thin \eqn{Q \in \bold{R}^{n \times p}}. If \eqn{r = \mathrm{rank}(X) < p}, then only the reduced \eqn{Q \in \mathbb{R}^{n \times r}} matrix is returned.
#' @return returns part or all of \eqn{Q}, the order-\eqn{n} orthogonal (unitary) transformation represented by qr. If complete is TRUE, \eqn{Q} has \eqn{n} columns. If complete is FALSE, \eqn{Q} has \eqn{p} columns.
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## get the full Q matrix
#' Q1 <- qr_Q(qr_res$qr, qr_res$qraux, complete = TRUE)
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q1)-diag(1, n)))
#'
#' ## get the reduced Q matrix
#' Q2 <- qr_Q(qr_res$qr, qr_res$qraux, qr_res$rank, complete = FALSE)
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q2)-diag(1, p)))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_Q <- function(qr, tau, rank = NULL, complete = NULL) {
    .Call(`_fastQR_qr_Q`, qr, tau, rank, complete)
}

#' @name qr_R
#' @title Reconstruct the R, matrix from a QR object.
#' @description returns the \eqn{R} matrix of the full QR decomposition. If \eqn{r = \mathrm{rank}(X) < p}, then only the reduced \eqn{R \in \mathbb{R}^{r \times p}} matrix is returned.
#' @param qr object representing a QR decomposition. This will typically have come from a previous call to qr.
#' @param rank the rank of x as computed by the decomposition.
#' @param pivot a logical value indicating whether to pivot the columns of \eqn{X}. Defaults to FALSE, meaning no pivoting is performed.
#' @param complete logical flag (length 1). Indicates whether the \eqn{R} matrix is to be completed by binding zero-value rows beneath the square upper triangle. If \eqn{r = \mathrm{rank}(X) < p}, then only the reduced \eqn{R \in \mathbb{R}^{r \times p}} matrix is returned.
#' @param pivot a vector of length \eqn{p}, specifying the permutation of the columns of \eqn{X} applied during the QR decomposition process. The default is NULL if no pivoting has been applied.
#' @return returns part or all of \eqn{R}. If complete is TRUE, \eqn{R} has \eqn{n} rows. If complete is FALSE, \eqn{R} has \eqn{p} rows.
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## get the full R matrix
#' R1 <- qr_R(qr_res$qr, complete = TRUE)
#'
#' ## check that X^TX = R^TR
#' ## get the permutation matrix
#' P <- qr_pivot2perm(pivot = qr_res$pivot)
#' max(abs(crossprod(R1 %*% P) - crossprod(X)))
#' max(abs(crossprod(R1) - crossprod(X %*% t(P))))
#'
#' ## get the reduced R matrix
#' R2 <- qr_R(qr_res$qr, qr_res$rank, complete = FALSE)
#'
#' ## check that X^TX = R^TR
#' ## get the permutation matrix
#' P <- qr_pivot2perm(pivot = qr_res$pivot)
#' max(abs(crossprod(R2 %*% P) - crossprod(X)))
#' max(abs(crossprod(R2) - crossprod(X %*% t(P))))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_R <- function(qr, rank = NULL, pivot = NULL, complete = NULL) {
    .Call(`_fastQR_qr_R`, qr, rank, pivot, complete)
}

#' @name qr_pivot2perm
#' @title Reconstruct the permutation matrix from the pivot vector.
#' @description returns the permutation matrix for the QR decomposition.
#' @param pivot a vector of dimension \eqn{n} of pivot elements from the QR factorization.
#' @return the perumutation matrix \eqn{P} of dimension \eqn{n \times n}.
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## get the pivot matrix
#' P <- qr_pivot2perm(qr_res$pivot)
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_pivot2perm <- function(pivot) {
    .Call(`_fastQR_qr_pivot2perm`, pivot)
}

#' @name qr_X
#' @title Reconstruct the original matrix from which the object was constructed \eqn{X\in\mathbb{R}^{n\times p}} from the Q and R matrices of the QR decomposition.
#' @description returns the \eqn{X\in\mathbb{R}^{n\times p}} matrix.
#' @param Q either the reduced \eqn{Q\in\mathbb{R}^{n\times p}} of full \eqn{Q\in\mathbb{R}^{n\times n}}, Q matrix obtained from the QR decomposition.
#' @param R either the reduced \eqn{R\in\mathbb{R}^{p\times p}} of full \eqn{R\in\mathbb{R}^{n\times p}}, R matrix obtained from the QR decomposition.
#' @param pivot a vector of length \eqn{p}, specifying the permutation of the columns of \eqn{X} applied during the QR decomposition process. The default is NULL if no pivoting has been applied.
#' @return returns the matrix \eqn{X}.
#'
#' @examples
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X, pivot = TRUE)
#'
#' ## get the Q and R matrices
#' Q  <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux, rank = qr_res$rank, complete = TRUE)
#' R  <- qr_R(qr = qr_res$qr, rank = qr_res$rank, complete = TRUE)
#' X1 <- qr_X(Q = Q, R = R, pivot = qr_res$pivot)
#' max(abs(X1 - X))
#'
#' ## get the full QR decomposition without pivot
#' qr_res <- fastQR::qr_fast(X = X, pivot = FALSE)
#'
#' ## get the Q and R matrices
#' Q  <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux, rank = p, complete = FALSE)
#' R  <- qr_R(qr = qr_res$qr, rank = NULL, complete = FALSE)
#' X1 <- qr_X(Q = Q, R = R, pivot = NULL)
#' max(abs(X1 - X))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_X <- function(Q, R, pivot = NULL) {
    .Call(`_fastQR_qr_X`, Q, R, pivot)
}

#' @name qr_Q_full
#' @title Reconstruct the full Q matrix from the qr object.
#' @description returns the full \eqn{Q\in\mathbb{R}^{n\times n}} matrix.
#' @param qr object representing a QR decomposition. This will typically have come from a previous call to qr.
#' @param tau a vector of length \eqn{ncol(X)} which contains additional information on \eqn{Q}. It corresponds to qraux from a previous call to qr.
#' @return returns the matrix \eqn{Q\in\mathbb{R}^{n\times n}}.
#'
#' @examples
#' ## create data: n > p
#' set.seed(1234)
#' n <- 12
#' p <- 7
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## complete the reduced Q matrix
#' Q <- fastQR::qr_Q_full(qr  = qr_res$qr,
#'                        tau = qr_res$qraux)
#'
#' ## check the Q matrix (orthogonality)
#' max(abs(crossprod(Q)-diag(1, n)))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_Q_full <- function(qr, tau) {
    .Call(`_fastQR_qr_Q_full`, qr, tau)
}

#' @name qr_Q_reduced2full
#' @title Reconstruct the full Q matrix from the reduced Q matrix.
#' @description returns the full \eqn{Q\in\mathbb{R}^{n\times n}} matrix.
#' @param Q a \eqn{n\times p} reduced Q matrix from the QR decomposition (with \eqn{n>p}).
#' @return a \eqn{n\times n} orthogonal matrix \eqn{Q}.
#'
#' @examples
#' ## create data: n > p
#' set.seed(1234)
#' n <- 12
#' p <- 7
#' X <- matrix(rnorm(n * p), n, p)
#'
#' ## get the full QR decomposition with pivot
#' qr_res <- fastQR::qr_fast(X = X,
#'                           tol = sqrt(.Machine$double.eps),
#'                           pivot = TRUE)
#'
#' ## reconstruct the reduced Q matrix
#' Q1 <- qr_Q(qr = qr_res$qr, tau = qr_res$qraux,
#'            rank = qr_res$rank, complete = FALSE)
#'
#' ## complete the reduced Q matrix
#' Q2 <- fastQR::qr_Q_reduced2full(Q = Q1)
#' R  <- fastQR::qr_R(qr = qr_res$qr, rank = NULL, complete = TRUE)
#'
#' X1 <- qr_X(Q = Q2, R = R, pivot = qr_res$pivot)
#' max(abs(X - X1))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
qr_Q_reduced2full <- function(Q) {
    .Call(`_fastQR_qr_Q_reduced2full`, Q)
}

#' @name qr_lm
#' @title Ordinary least squares for the linear regression model
#' @description qr_lm, or LS for linear regression models, solves the following optimization problem
#' \deqn{\textrm{min}_\beta ~ \frac{1}{2}\|y-X\beta\|_2^2,}
#' for \eqn{y\in\mathbb{R}^n} and \eqn{X\in\mathbb{R}^{n\times p}} witn \eqn{n>p}, to obtain a coefficient vector \eqn{\widehat{\beta}\in\mathbb{R}^p}. The design matrix \eqn{X\in\mathbb{R}^{n\times p}}
#' contains the observations for each regressor.
#' @param y a vector of length-\eqn{n} response vector.
#' @param X an \eqn{(n\times p)} full column rank matrix of predictors.
#' @param X_test an \eqn{(q\times p)} full column rank matrix. Test set. By default it set to NULL.
#' @return A named list containing \describe{
#' \item{coeff}{a length-\eqn{p} vector containing the solution for the parameters \eqn{\beta}.}
#' \item{coeff.se}{a length-\eqn{p} vector containing the standard errors for the estimated regression parameters \eqn{\beta}.}
#' \item{fitted}{a length-\eqn{n} vector of fitted values, \eqn{\widehat{y}=X\widehat{\beta}}.}
#' \item{residuals}{a length-\eqn{n} vector of residuals, \eqn{\varepsilon=y-\widehat{y}}.}
#' \item{residuals_norm2}{the squared L2-norm of the residuals, \eqn{\Vert\varepsilon\Vert_2^2.}}
#' \item{y_norm2}{the squared L2-norm of the response variable, \eqn{\Vert y\Vert_2^2.}}
#' \item{R}{the \eqn{R\in\mathbb{R}^{p\times p}} upper triangular matrix of the QR decomposition.}
#' \item{L}{the inverse of the \eqn{R\in\mathbb{R}^{p\times p}} upper triangular matrix of the QR decomposition \eqn{L = R^{-1}}.}
#' \item{XTX}{the Gram matrix \eqn{X^\top X\in\mathbb{R}^{p\times p}} of the least squares problem.}
#' \item{XTX_INV}{the inverse of the Gram matrix \eqn{X^\top X\in\mathbb{R}^{p\times p}} of the least squares problem \eqn{(X^\top X)^{-1}}.}
#' \item{XTy}{A vector equal to \eqn{X^\top y}, the cross-product of the design matrix \eqn{X} with the response vector \eqn{y}.}
#' \item{sigma2_hat}{An estimate of the error variance \eqn{\sigma^2}, computed as the residual sum of squares divided by the residual degrees of freedom \eqn{\widehat{\sigma}^2 = \frac{\|y - X\hat{\beta}\|_2^2}{df}}}
#' \item{df}{The residual degrees of freedom, given by \eqn{n - p}, where \eqn{n} is the number of observations and \eqn{p} is the number of estimated parameters.}
#' \item{R2}{\eqn{R^2}, coefficient of determination, measure of goodness-of-fit of the model.}
#' \item{predicted}{predicted values for the test set, \eqn{X_{\text{test}}\widehat{\beta}}. It is only available if X_test is not NULL.}
#' }
#' @examples
#'
#' ## generate sample data
#' ## create data: n > p
#' set.seed(1234)
#' n    <- 12
#' n0   <- 3
#' p    <- 7
#' X    <- matrix(rnorm(n * p), n, p)
#' b    <- rep(1, p)
#' sig2 <- 0.25
#' y    <- X %*% b + sqrt(sig2) * rnorm(n)
#' summary(lm(y~X))
#'
#' ## test
#' X_test <- matrix(rnorm(n0 * p), n0, p)
#'
#' ## lm
#' qr_lm(y = y, X = X, X_test = X_test)
#' qr_lm(y = y, X = X)
#'
qr_lm <- function(y, X, X_test = NULL) {
    .Call(`_fastQR_qr_lm`, y, X, X_test)
}

qr_lm_pred <- function(y, X, X_test) {
    .Call(`_fastQR_qr_lm_pred`, y, X, X_test)
}

#' @name rupdate
#' @title Fast updating of the R matrix
#' @description updates the R factorization when \eqn{m \geq 1} rows or columns are added to the matrix \eqn{X \in \mathbb{R}^{n \times p}}, where \eqn{n > p}. The R factorization of \eqn{X} produces an upper triangular matrix \eqn{R \in \mathbb{R}^{p \times p}} such that \eqn{X^\top X = R^\top R}. For more details on this method, refer to Golub and Van Loan (2013). Columns can only be added in positions \eqn{p+1} through \eqn{p+m}, while the position of added rows does not need to be specified.
#' @param X the current \eqn{n\times p} matrix, prior to the addition of any rows or columns.
#' @param R a \eqn{p\times p} upper triangular matrix.
#' @param U either a \eqn{n\times m} matrix or a \eqn{p\times m} matrix of columns or rows to be added.
#' @param type either 'row' of 'column', for adding rows or columns.
#' @param fast fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.
#' @return R the updated R matrix.
#'
#' @examples
#' ## Add one column
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create column to be added
#' u  <- matrix(rnorm(n), n, 1)
#' X1 <- cbind(X, u)
#'
#' ## update the R decomposition
#' R2 <- fastQR::rupdate(X = X, R = R1, U = u,
#'                       fast = FALSE, type = "column")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Add m columns
#' ## generate sample data
#' set.seed(1234)
#' n <- 10
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create the matrix of columns to be added
#' m  <- 2
#' U  <- matrix(rnorm(n*m), n, m)
#' X1 <- cbind(X, U)
#'
#' # QR update
#' R2 <- fastQR::rupdate(X = X, R = R1, U = U,
#'                       fast = FALSE, type = "column")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Add one row
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create the row u to be added
#' u  <- matrix(data = rnorm(p), p, 1)
#' k  <- 5
#' if (k<=n) {
#'   X1 <- rbind(rbind(X[1:(k-1), ], t(u)), X[k:n, ])
#' } else {
#'   X1 <- rbind(rbind(X, t(u)))
#' }
#'
#' ## update the R decomposition
#' R2 <- fastQR::rupdate(R = R1, X = X,
#'                       U = u,
#'                       type = "row")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Add m rows
#' ## generate sample data
#' set.seed(1234)
#' n <- 12
#' p <- 5
#' X <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## create the matrix of rows to be added
#' m  <- 2
#' U  <- matrix(data = rnorm(p*m), p, m)
#' k  <- 5
#' if (k<=n) {
#'   X1 <- rbind(rbind(X[1:(k-1), ], t(U)), X[k:n, ])
#' } else {
#'   X1 <- rbind(rbind(X, t(U)))
#' }
#'
#' ## update the R decomposition
#' R2 <- fastQR::rupdate(R = R1, X = X,
#'                       U = U,
#'                       fast = FALSE,
#'                       type = "row")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
NULL

rupdate <- function(X, R, U, fast = NULL, type = NULL) {
    .Call(`_fastQR_rupdate`, X, R, U, fast, type)
}

#' @name rdowndate
#' @title Fast downdating of the R matrix
#' @description rdowndate provides the update of the thin R matrix of the QR factorization after the deletion of \eqn{m\geq 1} rows or columns to the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The R factorization of the matrix \eqn{X} returns the upper triangular matrix \eqn{R\in\mathbb{R}^{p\times p}} such that \eqn{X^\top X=R^\top R}. See Golub and Van Loan (2013) for further details on the method.
#' @param R a \eqn{p\times p} upper triangular matrix.
#' @param k position where the columns or the rows are removed.
#' @param m number of columns or rows to be removed. It is not required when deleting columns. If NULL, it defaults to the number of columns in \eqn{U}.
#' @param U a \eqn{p\times m} matrix of rows to be removed. It should only be provided when rows are being removed.
#' @param type either 'row' of 'column', for removing rows or columns.
#' @param fast fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.
#' @return R the updated R matrix.
#'
#' @examples
#' ## Remove one column
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## select the column to be deleted from X and update X
#' k  <- 2
#' X1 <- X[, -k]
#'
#' ## downdate the R decomposition
#' R2 <- fastQR::rdowndate(R = R1, k = k,
#'                         m = 1, type = "column")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Remove m columns
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## select the column to be deleted from X and update X
#' k  <- 2
#' X1 <- X[, -c(k,k+1)]
#'
#' ## downdate the R decomposition
#' R2 <- fastQR::rdowndate(R = R1, k = k,
#'                         m = 2, type = "column")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Remove one row
#' ## generate sample data
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#'
#' ## get the initial QR factorization
#' output <- fastQR::qr(X, type = "householder",
#'                      nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' # select the row to be deleted from X and update X
#' k  <- 5
#' X1 <- X[-k,]
#' U  <- as.matrix(X[k,], p, 1)
#'
#' ## downdate the R decomposition
#' R2 <-  rdowndate(R = R1, k = k, m = 1,
#'                  U = U, fast = FALSE, type = "row")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' ## Remove m rows
#' ## create data: n > p
#' set.seed(10)
#' n      <- 10
#' p      <- 6
#' X      <- matrix(rnorm(n * p, 1), n, p)
#' output <- fastQR::qr(X, type = "householder",
#'                     nb = NULL,
#'                      complete = TRUE)
#' Q      <- output$Q
#' R      <- output$R
#' R1     <- R[1:p,]
#'
#' ## select the rows to be deleted from X and update X
#' k  <- 2
#' m  <- 2
#' X1 <- X[-c(k,k+m-1),]
#' U  <- t(X[k:(k+m-1), ])
#'
#' ## downdate the R decomposition
#' R2 <- rdowndate(R = R1, k = k, m = m,
#'                 U = U, fast = FALSE, type = "row")
#'
#' ## check
#' max(abs(crossprod(R2) - crossprod(X1)))
#'
#' @references
#' \insertRef{golub_van_loan.2013}{fastQR}
#'
#' \insertRef{bjorck.2015}{fastQR}
#'
#' \insertRef{bjorck.2024}{fastQR}
#'
#' \insertRef{bernardi_etal.2024}{fastQR}
#'
rdowndate <- function(R, k = NULL, m = NULL, U = NULL, fast = NULL, type = NULL) {
    .Call(`_fastQR_rdowndate`, R, k, m, U, fast, type)
}

set_diff <- function(x, y) {
    .Call(`_fastQR_set_diff`, x, y)
}

mat_slicing_byrow <- function(X, index) {
    .Call(`_fastQR_mat_slicing_byrow`, X, index)
}

mat_slicing_byrow2 <- function(X, index) {
    .Call(`_fastQR_mat_slicing_byrow2`, X, index)
}

mat_slicing_byrow3 <- function(X, index) {
    .Call(`_fastQR_mat_slicing_byrow3`, X, index)
}

