# R/data_loading.R

#' Load Brazilian Roads Data
#'
#' This function loads the roads data. By default, it loads a small "toy" dataset
#' included in the package for testing and examples. To use the full dataset
#' (all Brazilian roads), set \code{type = "full"}. The full dataset will be
#' downloaded from Zenodo upon the first request and cached locally.
#'
#' @param type Character. Options are "toy" (default) or "full".
#'
#' @return An sf object containing roads data.
#' @export
load_roads <- function(type = "toy") {
  if (type == "toy") {
    # Carrega o dado de exemplo (pequeno) que você vai criar
    file_path <- system.file("extdata", "roads_toy.rds", package = "ecoTolerance")
    if (file_path == "") {
      stop("Toy data not found. Please ensure package is installed correctly.")
    }
    roads <- readRDS(file_path)
    # Garante CRS
    if (sf::st_crs(roads)$epsg != 4326) {
      roads <- sf::st_transform(roads, crs = 4326)
    }
    return(roads)

  } else if (type == "full") {
    # Define local de cache seguro (padrão CRAN)
    cache_dir <- tools::R_user_dir("ecoTolerance", which = "cache")
    if (!dir.exists(cache_dir)) dir.create(cache_dir, recursive = TRUE)

    dest_file <- file.path(cache_dir, "rodovias_full.rds")

    # URL DO ZENODO (VOCÊ DEVE COLOCAR O SEU LINK AQUI)
    url_zenodo <- "https://zenodo.org/records/18614927/files/rodovias_full.rds?download=1"

    if (!file.exists(dest_file)) {
      message("Downloading full roads dataset from Zenodo (~XX MB)... this may take a while.")
      # Aumenta timeout para 5 min
      old_timeout <- getOption("timeout")
      options(timeout = max(300, old_timeout))
      on.exit(options(timeout = old_timeout))

      tryCatch({
        utils::download.file(url_zenodo, destfile = dest_file, mode = "wb")
      }, error = function(e) {
        stop("Failed to download roads data. Please check your internet connection.")
      })
    } else {
      message("Loading full roads dataset from local cache.")
    }

    roads <- readRDS(dest_file)
    if (sf::st_crs(roads)$epsg != 4326) {
      roads <- sf::st_transform(roads, crs = 4326)
    }
    return(roads)

  } else {
    stop("Invalid type. Choose 'toy' or 'full'.")
  }
}

#' Load Human Footprint GeoTIFF
#'
#' This function loads the human footprint data. By default, it loads a small "toy"
#' raster included in the package. To use the full Brazil raster, set \code{type = "full"}.
#'
#' @param type Character. Options are "toy" (default) or "full".
#'
#' @return A RasterLayer object containing human footprint data.
#' @export
load_human_footprint <- function(type = "toy") {
  if (type == "toy") {
    file_path <- system.file("extdata", "footprint_toy.tif", package = "ecoTolerance")
    if (file_path == "") stop("Toy data not found.")
    return(raster::raster(file_path))

  } else if (type == "full") {
    cache_dir <- tools::R_user_dir("ecoTolerance", which = "cache")
    if (!dir.exists(cache_dir)) dir.create(cache_dir, recursive = TRUE)

    dest_file <- file.path(cache_dir, "human_footprint_brazil.tif")

    # URL DO ZENODO (VOCÊ DEVE COLOCAR O SEU LINK AQUI)
    url_zenodo <- "https://zenodo.org/records/18614927/files/human_footprint_brazil.tif?download=1"

    if (!file.exists(dest_file)) {
      message("Downloading full footprint raster from Zenodo... this may take a while.")
      old_timeout <- getOption("timeout")
      options(timeout = max(300, old_timeout))
      on.exit(options(timeout = old_timeout))

      tryCatch({
        utils::download.file(url_zenodo, destfile = dest_file, mode = "wb")
      }, error = function(e) {
        stop("Failed to download footprint data.")
      })
    } else {
      message("Loading full footprint raster from local cache.")
    }

    return(raster::raster(dest_file))
  } else {
    stop("Invalid type. Choose 'toy' or 'full'.")
  }
}