\name{changepoint.influence-package}
\alias{changepoint.influence-package}
\alias{changepoint.influence}
\docType{package}
\title{
\packageTitle{changepoint.influence}
}
\description{
\packageDescription{changepoint.influence}
}
\details{

The DESCRIPTION file:
\packageDESCRIPTION{changepoint.influence}
\packageIndices{changepoint.influence}
The package allows users to input their data, segmentation and function used for the segmentation (and additional arguments) and the package calculates the influence of the data on the changepoint locations.

The influence() function is the first port of call to calculate the influence.  We provide two methods for influence detection, via "delete" and "outlier" options which respectively consider the effect of deleting a data point or making it an outlier.  Currently we provide this method for cpt objects (as generated by the "changepoint" package) but plan to extend this to other objects in the future.  Please add requests for objects to include to our github issues.

Users are encouraged to explore the documentation for the StabilityOverview() graphic, followed by the LocationStability() and ParameterStability() graphics for a more granual view, followed by the InfluenceMap() as the highest level of detail.
}
\author{
\packageAuthor{changepoint.influence}

Maintainer: \packageMaintainer{changepoint.influence}
}
\references{
Wilms I, Killick R, Matteson DS (2022) Graphical Influence Diagnostics for Changepoint Models, \emph{Journal of Computational and Graphical Statistics},  \bold{31:3}, 753--765 \emph{DOI: 10.1080/10618600.2021.2000873}}
\keyword{package}
\keyword{changepoint}
\keyword{influence}
\seealso{
\code{\link{influence-methods}},\code{\link{StabilityOverview}},
\code{\link{LocationStability}}, \code{\link{ParameterStability}}, \code{\link{InfluenceMap}}
}
\examples{
\donttest{
#### Load the data in the R package changepoint.influence ####
data("welldata")
welllog = welldata[1001:2000] # Extract the mid section of the data as analyzed in other papers
n = length(welllog)
var = NULL; for (i in 30:1000){var[i]=var(welllog[(i-29):i])} 
welllogs = welllog/sqrt(median(var, na.rm = TRUE)) 
# rescale the data to have unit variance across time, 
# note that there may still be changes in variance across the series.

#### Apply PELT to the welllog data ####
out.PELT = cpt.mean(welllogs, method = 'PELT')

#### Calculate the influence measures ####
welllogs.inf = influence(out.PELT) 
# the code extracts all the details of the original cpt.mean() function call 
# and uses these in the calculation of the influence for the modified data.

#### Stability Dashboards ####
StabilityOverview(welllogs, cpts(out.PELT), welllogs.inf, las = 1,ylab='Nuclear-Magnetic Response',
  legend.args=list(display=TRUE,x="bottomright",y=NULL,cex=1.5,bty="n",horiz=FALSE,xpd=FALSE))
# We can specify where the legend will sit in the graphic via the legend.args
# which are passed to the legend() function.  We can also include additional arguments
# to pass to the plotting such as las=1 here.

#### Location Stability plot ####
exp.seg=LocationStability(cpts(out.PELT), welllogs.inf, type = 'Difference', cpt.lwd = 4, las = 1)
# Note that if the expected segmentation is not provided, it will be calcuated and then
# returned so that the user can avoid calculating this again in other plot calls.

#### Parameter Stability plot ####
ParameterStability(welllogs.inf, original.mean = rep(param.est(out.PELT)$mean, 
  times=diff(c(0,out.PELT@cpts))), las = 1, ylab = 'Nuclear-Magnetic Response')
# Note that the original.mean argument is provided for each timepoint so is a length n vector.
}

#### Influence Map ####
\dontrun{
library(ggplot2)
welllogs.inf = influence(out.PELT, method = "delete")
InfluenceMap(cpts(out.PELT),welllogs.inf,data=welllogs,include.data=TRUE,
    ylab='Nuclear-Magnetic\n Response',
    ggops=theme(axis.text=element_text(size=15),axis.title=element_text(size=20),
      plot.title=element_text(size=25)))
# The InfluenceMap uses ggplot2 functions, thus you can add theme options via the ggops argument.
# Here we change the text sizes to ensure readable titles and labels for a report.

welllogs.inf = influence(out.PELT, method = "outlier")
InfluenceMap(cpts(out.PELT), welllogs.inf, data = welllogs, include.data = TRUE, 
    ylab='Nuclear-Magnetic\n Response')
}
}
