
<!-- README.md is generated from README.Rmd. Please edit that file -->

# cardargus <img src="man/figures/logo.svg" align="right" height="139" alt="" />

<!-- badges: start -->

[![CRAN
status](https://www.r-pkg.org/badges/version/cardargus)](https://CRAN.R-project.org/package=cardargus)
[![Lifecycle:
experimental](https://img.shields.io/badge/lifecycle-experimental-orange.svg)](https://lifecycle.r-lib.org/articles/stages.html#experimental)
<!-- badges: end -->

**cardargus** is an R package for creating informative SVG cards with
embedded styles, Google Fonts, badges, and logos. Cards are
self-contained SVG files, ideal for dashboards, reports, and
visualizations.

## Installation

You can install the development version of cardargus from
[GitHub](https://github.com/) with:

``` r
# install.packages("devtools")
devtools::install_github("castlab/cardargus")
```

## Basic Example

``` r
library(cardargus)
#> cardargus: Generate beautiful SVG cards with embedded styles
#> Use svg_card() to create cards
#> Use svg_to_png() or svg_to_png_chrome() to export to PNG
#> For best font support: setup_fonts()

# Create an informative card
card <- svg_card(
  font = "Jost",
  title = "FAR",
  
  badges_data = list(
    list(label = "Units",         value = "500",     color = "white"),
    list(label = "Federal Grant", value = "$100M",   color = "white"),
    list(label = "State Match",   value = "$80M",    color = "white")
  ),
  
  fields = list(
    list(
      list(label = "Project Name", value = "Boa Vista Residential",
           with_icon = icon_house())
    ),
    list(
      list(label = "Address", value = "123 Flower Street - Downtown")
    ),
    list(
      list(label = "City",   value = "Olinda"),
      list(label = "Region", value = "Pernambuco")
    ),
    list(
      list(label = "Developer",   value = "State Government"),
      list(label = "Contractor",  value = "ABC Construction"), 
      list(label = "Type", value = "PS")  # optional
    )
  ),
  
  bg_color    = "#FF9900",
  title_color = "#fff",
  label_color = "#fff",
  width = 450,
  
  # You can use bundled SVGs or any local file path
  logos = c(
    get_svg_path("seduh.svg"), 
    get_svg_path("morar_bem.svg")
  ),
  logos_height = 40,
  
  bottom_logos = c(
    get_svg_path("tree.svg"), 
    get_svg_path("gov_pe.svg")
  ),
  bottom_logos_height = 40,
  
  footer = paste0(
    "Source: SEDUH/PE on ", 
    format(Sys.time(), "%Y/%m/%d at %H:%M"))
)
```

``` r
include_card_png(card, dpi = 300, width = '50%')
```

<div class="cardargus-card" style="width:50%; max-width:100%;">

<img src="data:image/png;base64,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" alt="Card generated by cardargus" style="width:100%; height:auto;" />

</div>

``` r
# Save as SVG
save_svg(card, "my_card.svg")

# Convert to high-quality PNG
svg_to_png(card, "my_card.png", dpi = 300)
```

## Displaying Cards in R Markdown / Quarto

cardargus provides functions to display cards directly in your
documents:

``` r
# Display card as inline SVG (best quality)
include_card(card)

# Display card as PNG (better compatibility)
include_card_png(card, dpi = 150)

# Save and get path for knitr::include_graphics()
path <- save_card_for_knitr(card, "my_card", format = "png")
knitr::include_graphics(path)
```

For chunk-based workflows, register the cardargus knitr engine:

``` r
# In your setup chunk
register_knitr_engine()
```

Then use `cardargus` as a chunk engine:

```` markdown
```{cardargus}`
svg_card(title = "My Card", ...)
```
````

## Features

- 📦 **Self-contained SVG**: All styles and fonts embedded
- 🎨 **Customizable**: Colors, fonts, icons, and layouts
- 🏷️ **Badges**: Shields.io-style with dynamic colors and uniform height
- 🖼️ **Icons**: Built-in SVG icon library
- 📄 **Export**: High-quality PNG with transparent background
- 🔤 **Google Fonts**: Native support via showtext/sysfonts
- 📊 **R Markdown/Quarto**: Direct display functions

## Custom Cards

``` r
# Define badges
badges <- list(
  list(label = "Units", value = "500", color = "white"),
  list(label = "Status", value = "Active", color = "#4CAF50")
)

# Define fields with custom icon
fields <- list(
  list(
    list(label = "Project", value = "Housing Development")
  ),
  list(
    list(label = "City", value = "Maturéia"),
    list(label = "State", value = "Paraíba")
  )
)

# Create card with logos
card <- svg_card(
  title = "HOUSING",
  badges_data = badges,
  fields = fields,
  bg_color = "#2c3e50",
  title_color = "#ecf0f1",
  width = 200,
  logos = c(get_svg_path("morar_bem.svg")),
  logos_height = 40,
)
```

<div class="cardargus-card" style="width:25%; max-width:100%;">

<img src="data:image/png;base64,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" alt="Card generated by cardargus" style="width:100%; height:auto;" />

</div>

## Bundled SVGs

``` r
# List available SVGs
list_bundled_svgs()

# Get full path
get_svg_path("morar_bem.svg")
```

## Available Icons

``` r
# Built-in icons
icon_house()        # House (default)
icon_building()     # Building
icon_construction() # Construction
icon_map_pin()      # Location
icon_money()        # Money

# Or use your own SVG file
# with_icon = "/path/to/custom_icon.svg"
```

## Font Setup

For best font rendering:

``` r
# Setup Google Fonts (recommended)
setup_fonts()

# Or download fonts for offline use
install_fonts()

# Check font availability
font_available("Jost")
```

## Chrome Rendering (Recommended)

For **perfect font rendering** with Google Fonts, use headless Chrome:

``` r
# Check if Chrome is available
chrome_available()

# If Chrome is not installed, download it automatically (~150MB)
ensure_chrome(download = TRUE)

# Convert to PNG with Chrome (best quality)
svg_to_png_chrome(card, "my_card.png", dpi = 300)

# Convert to PDF (vector output)
svg_to_pdf_chrome(card, "my_card.pdf")

# In R Markdown / Quarto - force Chrome engine
include_card_png(card, dpi = 300, engine = "chrome")
```

Install `chromote` for Chrome support:

``` r
install.packages("chromote")
```

## Authors

- **Andre Leite** - *Maintainer* - <leite@castlab.org>
- **Hugo Vasconcelos** - <hugo.vasconcelos@ufpe.br>
- **Diogo Bezerra** - <diogo.bezerra@ufpe.br>

## License

MIT License - see [LICENSE](LICENSE) for details.
