% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectories_core.R
\name{detect_main_trajectories}
\alias{detect_main_trajectories}
\title{Detect main temporal trajectories in group-year DAG}
\usage{
detect_main_trajectories(
  groups_cumulative_trajectories,
  group,
  jaccard_min = 0.05,
  intra_min = 0.1,
  k_out = 2,
  alpha = 1,
  beta = 0.1,
  top_M = 5,
  min_len = 3,
  use_docs_per_group = TRUE
)
}
\arguments{
\item{groups_cumulative_trajectories}{List containing three components:
\itemize{
\item \code{groups_similarity}: Nested list with similarity data for each group,
containing edges with \code{from}, \code{to}, \code{weight} (Jaccard), and \code{documents}
\item \code{groups_attributes}: Nested list with node attributes for each group,
containing \code{quantity_papers}, \code{prop_tracked_intra_group}, \code{tracked_documents},
\code{PY.sd}, and \code{network_until}
\item \code{docs_per_group}: Data frame mapping group IDs to document IDs for
accurate unique document counting
}}

\item{group}{Character ID of the group to analyze (e.g., "component1_g01")}

\item{jaccard_min}{Minimum Jaccard similarity for edges (default: 0.05).
Higher values create sparser graphs with stronger connections.}

\item{intra_min}{Minimum proportion of tracked documents within group for
nodes (default: 0.10). Higher values filter out weaker nodes.}

\item{k_out}{Maximum number of outgoing edges to keep per node (default: 2).
Controls graph sparsity - lower values create simpler backbone structures.}

\item{alpha}{Weight for edge strength in path scoring (default: 1).
Higher values emphasize transition strength over node quality.}

\item{beta}{Per-step persistence bonus in path scoring (default: 0.1).
Higher values encourage longer trajectories.}

\item{top_M}{Maximum number of disjoint trajectories to extract (default: 5)}

\item{min_len}{Minimum number of distinct years for valid trajectory (default: 3)}

\item{use_docs_per_group}{Whether to use document IDs for accurate unique
document counting (default: TRUE). If FALSE, uses approximation.}
}
\value{
A list with two components:
\itemize{
\item \code{graph}: An igraph object representing the temporal DAG with scoring
attributes and optional document IDs
\item \code{trajectories}: A tibble of detected trajectories sorted by score, with columns:
\itemize{
\item \code{traj_id}: Trajectory identifier ("tr1", "tr2", ...)
\item \code{start}, \code{end}: First and last year of the trajectory
\item \code{length}: Number of distinct years in the trajectory
\item \code{nodes}: List of node names along the path (e.g., "y2009g03")
\item \code{score}: Total path score from dynamic programming
\item \code{mean_w}: Mean edge score along the path
\item \code{sum_docs}: Count of unique documents covered by the path
\item \code{mean_size}: Mean node size (quantity_papers × proportion tracked)
\item \code{mean_PYsd}: Mean publication year standard deviation
}
}
}
\description{
Identifies the most significant temporal trajectories within a group's evolution
over time by building a directed acyclic graph (DAG) from similarity data and
extracting highest-scoring disjoint paths using dynamic programming.
}
\details{
This function implements a comprehensive pipeline for detecting significant
temporal trajectories in research group evolution:
\subsection{Algorithm Overview}{
\enumerate{
\item \strong{Build Temporal DAG}: Constructs a directed acyclic graph where:
\itemize{
\item Nodes represent group-year combinations filtered by \code{intra_min} quality threshold
\item Edges represent transitions between consecutive years filtered by \code{jaccard_min}
\item Graph is sparsified to top \code{k_out} edges per node
}
\item \strong{Score Components}: Computes node and edge scores:
\itemize{
\item Node score: \eqn{s_v = \log(1 + \text{quantity\_papers}_v \times \text{prop\_tracked\_intra\_group}_v)}
\item Edge score: \eqn{s_e = \text{weight}_e \times \log(1 + \text{documents}_e)}
}
\item \strong{Extract Trajectories}: Uses dynamic programming to find heaviest paths:
\itemize{
\item Path score: \eqn{\text{best}(v) = \max\left( s_v, \max_{u \to v} \left( \text{best}(u) + s_v + \alpha \cdot s_{(u,v)} + \beta \right) \right)}
\item Iteratively extracts top \code{top_M} disjoint trajectories
\item Trajectories must span at least \code{min_len} distinct years
}
\item \strong{Count Documents}: Calculates unique document coverage:
\itemize{
\item If \code{use_docs_per_group = TRUE}: Exact count via set union of document IDs
\item Otherwise: Approximation: \eqn{\sum \text{node documents} - \sum \text{edge documents}}
}
}
}

\subsection{Parameter Tuning Guidance}{
\itemize{
\item For \strong{smoother, longer trajectories}: Increase \code{beta} (persistence bonus)
\item For \strong{transition-focused scoring}: Increase \code{alpha} (edge weight)
\item For \strong{denser connectivity}: Lower \code{jaccard_min} or increase \code{k_out}
\item For \strong{higher quality nodes}: Increase \code{intra_min}
\item For \strong{exact document counts}: Ensure \code{use_docs_per_group = TRUE} and provide \code{docs_per_group} data
}
}
}
\examples{
\dontrun{
# Basic usage with default parameters
trajectories <- detect_main_trajectories(
  groups_cumulative_trajectories = my_data,
  group = "component1_g01"
)

# Tuned for longer, transition-focused trajectories
trajectories <- detect_main_trajectories(
  groups_cumulative_trajectories = my_data,
  group = "component1_g01",
  jaccard_min = 0.03, # More permissive connectivity
  k_out = 3, # Denser backbone
  alpha = 1.5, # Emphasize edge strength
  beta = 0.2, # Encourage longer paths
  top_M = 8, # Extract more trajectories
  min_len = 4 # Require longer trajectories
)

# Access results
graph <- trajectories$graph
trajectory_data <- trajectories$trajectories

# Plot the top trajectory
top_trajectory <- trajectory_data[1, ]
}

}
\seealso{
\code{\link[=filter_trajectories]{filter_trajectories()}} for post-processing detected trajectories,
\code{\link[=plot_group_trajectories_lines_2d]{plot_group_trajectories_lines_2d()}} and \code{\link[=plot_group_trajectories_lines_3d]{plot_group_trajectories_lines_3d()}}
for visualization
}
