#' @title Core dimension
#' @description Given a game, this function computes the dimension of its core.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return The dimension of the core of \code{v}, as an integer.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @examples
#' v1 <- c(rep(0,5),rep(1,4),0,rep(1,3),2,2)
#' plotcoreset(v1)
#' coredimension(v1)
#'
#' v2 <- c(rep(0,5),rep(1,4),0,rep(1,4),2)
#' plotcoreset(v2)
#' coredimension(v2)
#'
#' v3 <- marginalgame(c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10),1)
#' plotcoreset(v3)
#' coredimension(v3)
#'
#' v4 <- c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' plotcoreset(v4)
#' coredimension(v4)
#' @references Edgeworth, F. Y. (1881). \emph{Mathematical psychics: An essay on the application of mathematics to the moral sciences}. CK Paul.
#' @references Gillies, D. (1953). \emph{Some theorems on n-person games}. PhD thesis, Princeton, University Press Princeton, New Jersey.
#' @seealso \link{balancedcheck}, \link{corevertices}, \link{corevertices234}, \link{plotcoreset}, \link{plotcoresets}
#' @export

coredimension <- function(v, binary = FALSE, tol = 1e-12) {

  V <- corevertices(v,binary=binary) # V: matriz con puntos como filas (cada fila es un vértice)
  if (is.null(V)) {
    # No hace falta poner mensaje, ya nos llega uno al hacer corevertices(v).
    return(NULL)
  }

  # svd(direction_vectors) da problemas cuando el core tiene un único punto.
  # Así que vamos a tratar primero los casos chorras por separado.
  if (dim(V)[1] == 1) { # si el core tiene un único vértice...
    message("The core of 'v' is made of a single point.")
    return(0)
  } else if (dim(V)[1] == 2) { # si el core tiene dos vértices...
    message("The core of 'v' is the segment between two points.")
    return(1)
  }

  # Si llegamos hasta aquí, es porque el core tiene al menos tres vértices.

  base_point <- V[1, , drop = FALSE] # Tomamos como base el primer punto
  direction_vectors <- sweep(V[-1, , drop = FALSE], 2, base_point) # Calculamos los vectores de dirección respecto al primer punto

  # La dimensión es el rango de la matriz de vectores de dirección
  # Dejamos sin ejecutar la próxima línea, fallos numéricos con los vectores.
  # r <- qr(direction_vectors, tol = tol)$rank # Factorización QR
  # En vez de lo anterior hacemos lo siguiente:
  s <- svd(direction_vectors)
  cd <- sum(s$d > tol)  # Número de valores singulares significativamente distintos de 0
  cd <- as.integer(cd)

  return(cd)

}
