#' @title Belong to core
#' @description This function checks if an allocation belongs to the core of a game.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param x An allocation, as a vector.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if \code{x} belongs to the core of \code{v}, \code{FALSE} otherwise. If \code{instance=TRUE} and \code{x} does not belong to the core of \code{v}, a justification is also provided: if efficiency is violated, \code{not efficient} is returned; if efficiency is not violated, the position (binary order position if \code{binary=TRUE}; lexicographic order position otherwise) of a coalition for which rationality is violated is returned.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @examples
#' v <- c(0, 0, 0, 2, 1, 4, 6)
#' a <- c(3, 1, 2) # an allocation for v
#' b <- c(2, 2, 2) # egalitarian solution for v
#' belong2corecheck(v = v, binary = TRUE, x = a, instance = TRUE)
#' belong2corecheck(v = v, binary = FALSE, x = b, instance = TRUE)
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # cost game
#' cost.x <- c(1,2,2) # core allocation of cost.v
#' belong2corecheck(v = -cost.v, x = -cost.x)
#' @references Gillies, D. (1953). \emph{Some theorems on n-person games}. PhD thesis, Princeton, University Press Princeton, New Jersey.
#' @export

belong2corecheck <- function(v, binary = FALSE, x, instance = FALSE) {

  ##################################
  ### Comprobación datos entrada ###
  ##################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (length(x) != n) {
    stop("'x' must have length n.")
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  S <- 1 # Primera coalición
  tol <- 100*.Machine$double.eps

  ################################
  ###### Cuerpo de la función######
  ################################

  # Condicion de eficiencia:
  if (abs(sum(x) - v[nC]) > tol) {
    checkR <- FALSE # Si x(N)!=v(N), x no es interior al core.
    exampleR <- "not efficient" # Falla la condición de eficiencia. Devuelve la coalición el número de la coalición N.
  } else { # Condicion de estabilidad:
    while (checkR & S <= nC - 1) {
      num <- S # Primero calculo el vector con los jugadores pertenecientes a la coalición en posición jj.
      J <- floor(log2(num)) + 1
      Sjug <- c() # Inicializo la coalición S

      for (ii in 1:J)
      {
        if (num %% 2 == 1) {
          Sjug <- c(Sjug, ii)
        }
        num <- num %/% 2
      }

      sumxS <- sum(x[Sjug]) # x(S)
      if (sumxS - v[S] + tol < 0) {
        checkR <- FALSE # Si x(S)<v(S), x no es interior al core.
        exampleR <- S # código numérico de una coalición donde falla la condición.
      }
      S <- S + 1
    }
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también el código numérico de la coalición S.
    if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, devolvemos la posición lexicográfico de una coalición que falla.
      return(list(check = checkR, example = codebin2lex(n, exampleR)))
    } else { # Si el juego se introdujo en binario, devolvemos la posición binaria de una coalición que falla.
      return(list(check = checkR, example = exampleR))
    }
  }
} # Fin de la función
