% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ExWALD.R
\name{ExWALD}
\alias{ExWALD}
\title{The Ex-Wald family}
\usage{
ExWALD(mu.link = "log", sigma.link = "log", nu.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "log" link as the default for the sigma parameter.}

\item{nu.link}{defines the nu.link, with "log" link as the default for the nu parameter.}
}
\value{
Returns a gamlss.family object which can be used to fit a 
Ex-WALD distribution in the \code{gamlss()} function.
}
\description{
The function \code{ExWALD()} defines the Ex-wALD distribution, three-parameter
continuous distribution for a \code{gamlss.family} object to be used in 
GAMLSS fitting using the function \code{gamlss()}.
}
\details{
The Ex-Wald distribution with parameters \eqn{\mu}, \eqn{\sigma} 
and \eqn{\nu} has density given by

\eqn{f(x |\mu, \sigma, \nu) = \frac{1}{\nu} \exp(\frac{-x}{\nu} + \sigma(\mu-k)) F_W(x|k, \sigma) \, \text{for} \, k \geq 0}

\eqn{f(x |\mu, \sigma, \nu) = \frac{1}{\nu} \exp\left( \frac{-(\sigma-\mu)^2}{2x} \right) Re \left( w(k^\prime \sqrt{x/2} + \frac{\sigma i}{\sqrt{2x}}) \right)  \, \text{for} \, k < 0}

where \eqn{k=\sqrt{\mu^2-\frac{2}{\nu}}}, 
\eqn{k^\prime=\sqrt{\frac{2}{\nu}-\mu^2}} and
\eqn{F_W} corresponds to the cumulative function of 
the Wald distribution. 

More details about those expressions
can be found on page 680 from Heathcote (2004).
}
\examples{
# Example 1
# Generating random values with
# known mu, sigma and nu

mu <- 0.20
sigma <- 70
nu <- 115

set.seed(123)
y <- rExWALD(n=100, mu, sigma, nu)

library(gamlss)
mod1 <- gamlss(y~1, family=ExWALD,
               control=gamlss.control(n.cyc=1000, trace=TRUE))

exp(coef(mod1, what="mu"))
exp(coef(mod1, what="sigma"))
exp(coef(mod1, what="nu"))

# Example 2
# Generating random values under some model

\donttest{
# A function to simulate a data set with Y ~ ExWALD
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu    <- exp(-1 + 2.8 * x1) # 1.5 approximately
  sigma <- exp( 1 - 1.2 * x2) # 1.5 approximately
  nu    <- 2
  y <- rExWALD(n=n, mu=mu, sigma=sigma, nu=nu)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(1234)
dat <- gendat(n=200)

# Fitting the model
mod2 <- gamlss(y ~ x1,
               sigma.fo = ~ x2,
               nu.fo = ~ 1,
               family = ExWALD,
               data = dat,
               control = gamlss.control(n.cyc=1000, 
                                        trace=TRUE))
summary(mod2)
}
}
\references{
Schwarz, W. (2001). The ex-Wald distribution as a descriptive model 
of response times. Behavior Research Methods, 
Instruments, & Computers, 33, 457-469.

Heathcote, A. (2004). Fitting Wald and ex-Wald distributions to 
response time data: An example using functions for the S-PLUS package. 
Behavior Research Methods, Instruments, & Computers, 36, 678-694.
}
\seealso{
\link{dExWALD}.
}
\author{
Freddy Hernandez, \email{fhernanb@unal.edu.co}
}
