################################################################################
# Copyright 2016 Indiana University
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
################################################################################

#' Runs all of the sparse matrix microbenchmarks
#'
#' \code{RunSparseMatrixBenchmark} runs all of the microbenchmarks for
#'   performance testing the sparse matrix linear algebra kernels
#'
#' This function runs the sparse matrix microbenchmarks, which are divided
#' into four categories supported by this benchmark, defined in the
#' \code{matrixVectorMicrobenchmarks}, \code{choleskyMicrobenchmarks}, 
#' \code{luMicrobenchmarks}, and \code{qrMicrobenchmarks} input lists
#' For each microbenchmark, it attempts to create a separate output file in CSV
#' format containing the performance results for each matrix tested by the
#' microbenchmark.  The names of the output files follow the format
#' \code{benchmarkName}_\code{runIdentifier}.csv, where
#' \code{benchmarkName} is specified in the
#' \code{SparseMatrixMicrobenchmark} object of each microbenchmark and
#' \code{runIdentifier} is an input parameter to this function.  If the file,
#' already exists, the results will be appended to the existing file.  Each
#' input lists contains instances of the
#' \code{\link{SparseMatrixMicrobenchmark}} class defining each microbenchmark.
#' Each microbenchmark object with the
#' \code{active} field set to TRUE will be executed.  The lists of default
#' microbenchmarks are generated by the functions
#' \code{\link{GetSparseMatrixVectorDefaultMicrobenchmarks}},
#' \code{\link{GetSparseCholeskyDefaultMicrobenchmarks}},
#' \code{\link{GetSparseLuDefaultMicrobenchmarks}}, and
#' \code{\link{GetSparseQrDefaultMicrobenchmarks}}.  Each 
#' \code{SparseMatrixMicrobenchmark} specifies an R data file which contains the
#' sparse matrix object needed by the microbenchmark.  The needed R data files
#' should either be given in an attached R package or given in the \code{data}
#' subdirectory of the current working directory, and they should have the
#' extension \code{.RData}.  If the linear algebra kernels are multithreaded,
#' by linking to multithreaded BLAS or LAPACK libraries for example, then the
#' number of threads must be retrievable from an environment variable which is
#' set before execution of the R programming environment.  The name of the
#' environment variable specifying the number of threads must be provided in
#' the R HPC benchmark environment variable R_BENCH_NUM_THREADS_VARIABLE.  This
#' function will retrieve the number of threads through
#' R_BENCH_NUM_THREADS_VARIABLE so that the number of threads can be printed to
#' the results files and recorded in data frames for reporting purposes.  This
#' function utilizes the number of threads only for reporting purposes and is
#' not used by the benchmark to effect the actual number of threads utilized by
#' the kernels, as that is assumed to be controlled by the numerical library.
#' An error exception will be thrown if the environment variable
#' R_BENCH_NUM_THREADS_VARIABLE and the variable it is set to are not both set.
#'
#' @param runIdentifier a character string specifying the suffix to be
#'   appended to the base of the file name of the output CSV format files
#' @param resultsDirectory a character string specifying the directory
#'   where all of the CSV performance results files will be saved
#' @param matrixVectorMicrobenchmarks a list of
#'   \code{SparseMatrixMicrobenchmark} objects defining the matrix-vector
#'   multiplication microbenchmarks to execute as part of the sparse matrix
#'   benchmark.  Default values are provided by the function
#'   \code{\link{GetSparseMatrixVectorDefaultMicrobenchmarks}}.  If the
#'   value is NULL, then all of the matrix-vector multiplication
#'   microbenchmarks will be skipped.
#' @param choleskyMicrobenchmarks a list of
#'   \code{SparseMatrixMicrobenchmark} objects defining the Cholesky
#'   factorization microbenchmarks to execute as part of the sparse matrix
#'   benchmark.  Default values are provided by the function
#'   \code{\link{GetSparseCholeskyDefaultMicrobenchmarks}}.  If the value
#'   is NULL, then all of the Cholesky factorization microbenchmarks will
#'   be skipped.
#' @param luMicrobenchmarks a list of \code{SparseMatrixMicrobenchmark}
#'   objects defining the LU factorization microbenchmarks to execute as part
#'   of the sparse matrix benchmark.  Default values are provided by the
#'   function \code{\link{GetSparseLuDefaultMicrobenchmarks}}.  If the value
#'   is NULL, then all of the LU factorization microbenchmarks will
#'   be skipped.
#' @param qrMicrobenchmarks a list of \code{SparseMatrixMicrobenchmark}
#'   objects defining the QR factorization microbenchmarks to execute as part
#'   of the sparse matrix benchmark.  Default values are provided by the
#'   function \code{\link{GetSparseQrDefaultMicrobenchmarks}}.  If the value
#'   is NULL, then all of the QR factorization microbenchmarks will
#'   be skipped.
#' @return a data frame containing the benchmark name, user, system, and
#'   elapsed (wall clock) times of each performance trial for each
#'   microbenchmark
#'   
#' @examples 
#' \dontrun{
#' # Set needed environment variables for multithreading.  Only single threading
#' # is used in the example.
#' #
#' # Note: The environment variables are usually set by the user before starting
#' #       the R programming environment; they are set here only to facilitate
#' #       a working example.  See the section on multithreading in the vignette
#' #       for further details.
#' Sys.setenv(R_BENCH_NUM_THREADS_VARIABLE="MKL_NUM_THREADS")
#' Sys.setenv(MKL_NUM_THREADS="1")
#' #
#' # Generate example microbenchmarks that can be run in a few minutes; see
#' # the vignette for more involved examples.  The matvec_laplacian7pt_100
#' # and cholesky_ct20stif microbenchmarks are defined in the examples.
#' #
#' # Note: The example microbenchmarks are different than the microbenchmarks
#' #       generated by
#' #       \code{\link{GetSparseMatrixVectorDefaultMicrobenchmarks}},
#' #       \code{\link{GetSparseCholeskyDefaultMicrobenchmarks}},
#' #       \code{\link{GetSparseLuDefaultMicrobenchmarks}}, and
#' #       \code{\link{GetSparseQrDefaultMicrobenchmarks}};
#' #       they were chosen for their short run times and suitability for
#' #       example code. 
#' exampleMatrixVectorMicrobenchmarks <- GetSparseMatrixVectorExampleMicrobenchmarks()
#' exampleCholeskyMicrobenchmarks <- GetSparseCholeskyExampleMicrobenchmarks()
#' # Set the output directory of the CSV summary results files
#' resultsDirectory <- "./SparseMatrixExampleOutput"
#' # Create the output directory
#' dir.create(resultsDirectory)
#' # Set an appropriate run identifier
#' runIdentifier <- "example"
#' # Run only the matrix-vector and Cholesky factorization microbenchmarks, as
#' # the others take a long time
#' resultsFrame <- RunSparseMatrixBenchmark(runIdentifier, resultsDirectory,
#'    matrixVectorMicrobenchmarks=exampleMatrixVectorMicrobenchmarks,
#'    choleskyMicrobenchmarks=exampleCholeskyMicrobenchmarks,
#'    luMicrobenchmarks=NULL,
#'    qrMicrobenchmarks=NULL)
#'
#' # This example runs only the Cholesky factorization microbenchmarks.
#' runIdentifier <- "choleksy_only"
#' # Run only the sparse Choleksy factorization microbenchmarks
#' choleskyResults <- RunSparseMatrixBenchmark(runIdentifier, resultsDirectory,
#'    matrixVectorMicrobenchmarks=NULL, luMicrobenchmarks=NULL,
#'    qrMicrobenchmarks=NULL)
#'
#' }
#'
#' @seealso \code{\link{GetSparseMatrixVectorDefaultMicrobenchmarks}}
#'          \code{\link{GetSparseCholeskyDefaultMicrobenchmarks}}
#'          \code{\link{GetSparseLuDefaultMicrobenchmarks}}
#'          \code{\link{GetSparseQrDefaultMicrobenchmarks}}
#'          \code{\link{GetSparseMatrixVectorExampleMicrobenchmarks}}
#'          \code{\link{GetSparseCholeskyExampleMicrobenchmarks}}
#' @export
RunSparseMatrixBenchmark <- function(runIdentifier,
   resultsDirectory,
   matrixVectorMicrobenchmarks = GetSparseMatrixVectorDefaultMicrobenchmarks(),
   choleskyMicrobenchmarks = GetSparseCholeskyDefaultMicrobenchmarks(),
   luMicrobenchmarks = GetSparseLuDefaultMicrobenchmarks(),
   qrMicrobenchmarks = GetSparseQrDefaultMicrobenchmarks()) {

   numberOfThreads <- GetNumberOfThreads()

#   allResults <- list()
   allResults <- NULL
   matrixVectorResults <- NULL
   choleskyResults <- NULL
   luResults <- NULL
   qrResults <- NULL

   # Loop over all sparse matrix-vector multiplication microbenchmarks
   if (length(matrixVectorMicrobenchmarks) > 0) {
      matrixVectorResults <- PerformSparseMatrixKernelMicrobenchmarking(matrixVectorMicrobenchmarks, numberOfThreads, runIdentifier, resultsDirectory)
   } else {
      cat(sprintf("WARN: no sparse matrix-vector multiplication microbenchmarks to execute, skipping\n\n"))
   }

   # Loop over all sparse Cholesky factorization microbenchmarks
   if (length(choleskyMicrobenchmarks) > 0) {
      choleskyResults <- PerformSparseMatrixKernelMicrobenchmarking(choleskyMicrobenchmarks, numberOfThreads, runIdentifier, resultsDirectory)
   } else {
      cat(sprintf("WARN: no sparse Cholesky factorization microbenchmarks to execute, skipping\n\n"))
   }

   # Loop over all sparse LU factorization microbenchmarks
   if (length(luMicrobenchmarks) > 0) {
      luResults <- PerformSparseMatrixKernelMicrobenchmarking(luMicrobenchmarks, numberOfThreads, runIdentifier, resultsDirectory)
   } else {
      cat(sprintf("WARN: no sparse LU factorization microbenchmarks to execute, skipping\n\n"))
   }

   # Loop over all sparse QR factorization microbenchmarks
   if (length(qrMicrobenchmarks)) {
      qrResults <- PerformSparseMatrixKernelMicrobenchmarking(qrMicrobenchmarks, numberOfThreads, runIdentifier, resultsDirectory)
   } else {
      cat(sprintf("WARN: no sparse QR factorization microbenchmarks to execute, skipping\n\n"))
   }

   allResults <- rbind(allResults, matrixVectorResults)
   allResults <- rbind(allResults, choleskyResults)
   allResults <- rbind(allResults, luResults)
   allResults <- rbind(allResults, qrResults)

   return(allResults)
}


#' Performs microbenchmarking of sparse matrix kernels specified by an input
#' list 
#' 
#' \code{PerformSparseMatrixKernelMicrobenchmarking} performs microbenchmarking
#' of sparse matrix kernels specified by the input list of
#' \code{SparseMatrixMicrobenchmark} objects.  Objects with the \code{active}
#' flag set to TRUE indicate that the corresponding microbenchmark will be
#' performed; FALSE indicates that the microbenchmark will be skipped.
#' If the \code{matrixObjectName} field of an input
#' \code{SparseMatrixMicrobenchmark} object is set to \code{NA_character_},
#' then the sparse matrix is assumed to be dynamically generated by the
#' allocator function specified in the \code{allocatorFunction} field.
#' If the \code{matrixObjectName} field is specified, then the sparse
#' matrix object is expected to be found in an \code{.RData} file with
#' base file name the same as the value of \code{matrixObjectName}, and
#' located in the either an attached R data package or a directory named
#' \code{data} in the current working directory.  See the
#' the \code{\link[utils]{data}} package for more details.
#'
#' @param microbenchmarks a list of
#'   \code{SparseMatrixMicrobenchmark} objects defining the sparse matrix
#'   microbenchmarks to be executed as part of the sparse matrix
#'   benchmark.
#' @param numberOfThreads the number of threads the microbenchmarks are
#'   intended to be executed with; the value is for display purposes only as
#'   the number of threads used is assumed to be controlled through environment
#'   variables
#' @param runIdentifier a character string specifying the suffix to be
#'   appended to the base of the file name of the output CSV format files
#' @param resultsDirectory a character string specifying the directory
#'   where all of the CSV performance results files will be saved
#' @return a data frame containing the benchmark name, user, system, and
#'   elapsed (wall clock) times of each performance trial for each
#'   microbenchmark
#' @seealso \code{\link[utils]{data}}
PerformSparseMatrixKernelMicrobenchmarking <- function(microbenchmarks,
   numberOfThreads, runIdentifier, resultsDirectory) {

   microbenchmarkResults <- NULL

   if (length(microbenchmarks) > 0) {
      for (i in 1:length(microbenchmarks)) {
         if (microbenchmarks[[i]]$active) {

            benchmarkName <- microbenchmarks[[i]]$benchmarkName
            matrixObjectName <- microbenchmarks[[i]]$matrixObjectName

            # The matrices are read in to the global environment so that they
            # only have to be read from storage once.
            loadSuccessful <- tryCatch({
               if (!is.na(matrixObjectName)) {
                  utils::data(list=c(matrixObjectName))
               }

               TRUE
            }, warning = function(war) {
               msg <- sprintf("ERROR: data() threw a warning -- %s", war)
               write(msg, stderr())
               return(FALSE)
            }, error = function(err) {
               msg <- sprintf("ERROR: data() threw an error -- %s", err)
               write(msg, stderr())
               return(FALSE)
            })

            if (loadSuccessful) {
               resultsFrame <- MicrobenchmarkSparseMatrixKernel(microbenchmarks[[i]], numberOfThreads, resultsDirectory, runIdentifier)
               microbenchmarkResults <- rbind(microbenchmarkResults, resultsFrame)
              
               if (!is.na(matrixObjectName)) {
                  remove(list=c(matrixObjectName), envir=.GlobalEnv)
               }

               invisible(gc())
            } else {
               microbenchmarkResults[[benchmarkName]] <- NULL
               msg <- sprintf("ERROR: data() failed to read sparse matrix '%s', skipping microbenchmark '%s'",
                  microbenchmarks[[i]]$matrixObjectName,
                  microbenchmarks[[i]]$benchmarkName)
               write(msg, stderr())
            }
         }
      }
   }

   return (microbenchmarkResults)
}


#' Initializes the list of default sparse matrix-vector microbenchmarks
#'
#' \code{GetSparseMatrixVectorDefaultMicrobenchmarks} defines the default sparse
#' matrix-vector microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The current sparse
#' matrix-vector microbenchmarks cover a variety of matrices of different
#' dimensions and number of non-zero values.  They are as follows:
#' \enumerate{
#'   \item matvec_laplacian7pt_100 -- 100x100x100 7-point Laplacian operator
#'   \item matvec_laplacian7pt_200 -- 200x200x200 7-point Laplacian operator
#'   \item matvec_ca2010 -- DIMACS10/ca2010 710145x710145 undirected graph
#'      matrix
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmarks appear in the order
#'   listed in the function description and are assigned the names enumerate
#'   in the description.
#' @seealso \code{\link{SparseMatrixMicrobenchmark}}
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseMatrixVectorDefaultMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["matvec_laplacian7pt_100"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "matvec_laplacian7pt_100",
      benchmarkDescription = "sparse matrix-vector mult. with 100x100x100 7-point Laplacian operator",
      matrixObjectName = "laplacian7pt_100",
      numberOfRows = as.integer(1000000),
      numberOfColumns = as.integer(1000000),
      numberOfNonzeros = as.integer(6940000),
      numberOfTrials = as.integer(c(20)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseMatrixVectorAllocator,
      benchmarkFunction = SparseMatrixVectorMicrobenchmark
   )

   microbenchmarks[["matvec_laplacian7pt_200"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "matvec_laplacian7pt_200",
      benchmarkDescription = "Sparse matrix-vector mult. with 200x200x200 7-point Laplacian operator",
      matrixObjectName = "laplacian7pt_200",
      numberOfRows = as.integer(8000000),
      numberOfColumns = as.integer(8000000),
      numberOfNonzeros = as.integer(55760000),
      numberOfTrials = as.integer(c(20)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseMatrixVectorAllocator,
      benchmarkFunction = SparseMatrixVectorMicrobenchmark
   )   

   microbenchmarks[["matvec_ca2010"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "matvec_ca2010",
      benchmarkDescription = "Sparse matrix-vector mult. with undirected weighted graph matrix ca2010 from the University of Florida Sparse Matrix Collection DIMACS10 matrix group",
      matrixObjectName = "ca2010",
      numberOfRows = as.integer(710145),
      numberOfColumns = as.integer(710145),
      numberOfNonzeros = as.integer(3489366),
      numberOfTrials = as.integer(c(20)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseMatrixVectorAllocator,
      benchmarkFunction = SparseMatrixVectorMicrobenchmark
   )

   return (microbenchmarks)
}


#' Initializes the list of default sparse Cholesky factorization
#' microbenchmarks
#'
#' \code{GetSparseCholeskyDefaultMicrobenchmarks} defines the default sparse
#' Cholesky factorization microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The current sparse
#' Cholesky factorization microbenchmarks cover a variety of matrices of
#' different dimensions and number of non-zero values.  They are as follows:
#' \enumerate{
#'   \item cholesky_ct20stif -- Boeing structural matrix with 2600295 nonzeros
#'   \item cholesky_Andrews -- computer vision matrix with 760154
#'   \item cholesky_G3_circuit -- AMD circuit simulation matrix with 7660826
#'      nonzeros
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmarks appear in the order
#'   listed in the function description and are assigned the names enumerated
#'   in the description.
#' @seealso \code{\link{SparseMatrixMicrobenchmark}}
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseCholeskyDefaultMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["cholesky_ct20stif"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "cholesky_ct20stif",
      benchmarkDescription = "Cholesky factorization of ct20stif matrix from University of Florida Sparse Matrix Collection Boeing group; CT20 engine block structural problem -- stiffness matrix, Boeing",
      matrixObjectName = "ct20stif",
      numberOfRows = as.integer(52329),
      numberOfColumns = as.integer(52329),
      numberOfNonzeros = as.integer(2600295),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseCholeskyAllocator,
      benchmarkFunction = SparseCholeskyMicrobenchmark
   )

   microbenchmarks[["cholesky_Andrews"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "cholesky_Andrews",
      benchmarkDescription = "Cholesky factorization of Andrews matrix from University of Florida Sparse Matrix Collection Andrews group; Eigenvalue problem from computer vision/graphics, Stuart Andrews, Brown Univ.",
      matrixObjectName = "Andrews",
      numberOfRows = as.integer(60000),
      numberOfColumns = as.integer(60000),
      numberOfNonzeros = as.integer(760154),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseCholeskyAllocator,
      benchmarkFunction = SparseCholeskyMicrobenchmark
   )

   microbenchmarks[["cholesky_G3_circuit"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "cholesky_G3_circuit",
      benchmarkDescription = "Cholesky factorization of G3_circuit matrix from University of Florida Sparse Matrix Collection AMD group; circuit simulation problem, Ufuk Okuyucu, AMD, Inc.",
      matrixObjectName = "G3_circuit",
      numberOfRows = as.integer(1585478),
      numberOfColumns = as.integer(1585478),
      numberOfNonzeros = as.integer(7660826),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseCholeskyAllocator,
      benchmarkFunction = SparseCholeskyMicrobenchmark
   )

   return (microbenchmarks)
}


#' Initializes the list of default sparse LU factorization microbenchmarks
#'
#' \code{GetSparseLuDefaultMicrobenchmarks} defines the default sparse LU
#' factorization microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The current sparse LU
#' factorization microbenchmarks cover a variety of matrices of different
#' dimensions and number of non-zero values.  They are as follows:
#' \enumerate{
#'   \item lu_circuit5M_dc -- Freescale DC circuit simulation matrix 2600295
#'     nonzeros
#'   \item lu_stomach -- 3D electro-physical model matrix with 3021648 nonzeros
#'   \item lu_torso3 -- 3D electro-physical model matrix with 4429042 nonzeros
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmarks appear in the order
#'   listed in the function description and are assigned the names enumerated
#'   in the description.
#' @seealso \code{\link{SparseMatrixMicrobenchmark}}
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseLuDefaultMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["lu_circuit5M_dc"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "lu_circuit5M_dc",
      benchmarkDescription = "LU decomposition of circuit5M_dc matrix from University of Florida Sparse Matrix Collection Freescale group; Large circuit (DC analysis) K. Gullapalli, Freescale Semiconductor",
      matrixObjectName = "circuit5M_dc",
      numberOfRows = as.integer(3523317),
      numberOfColumns = as.integer(3523317),
      numberOfNonzeros = as.integer(14865409),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseLuAllocator,
      benchmarkFunction = SparseLuMicrobenchmark
   )

   microbenchmarks[["lu_stomach"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "lu_stomach",
      benchmarkDescription = "LU decomposition of stomach matrix from University of Florida Sparse Matrix Collection Norris group; S.Norris, Univ. Auckland. 3D electro-physical model of a duodenum",
      matrixObjectName = "stomach",
      numberOfRows = as.integer(213360),
      numberOfColumns = as.integer(213360),
      numberOfNonzeros = as.integer(3021648),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseLuAllocator,
      benchmarkFunction = SparseLuMicrobenchmark
   )

   microbenchmarks[["lu_torso3"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "lu_torso3",
      benchmarkDescription = "LU decomposition of torso3 matrix from University of Florida Sparse Matrix Collection Norris group; S.Norris, Univ Auckland. finite diff. electro-phys.  3D model of torso",
      matrixObjectName = "torso3",
      numberOfRows = as.integer(259156),
      numberOfColumns = as.integer(259156),
      numberOfNonzeros = as.integer(4429042),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseLuAllocator,
      benchmarkFunction = SparseLuMicrobenchmark
   )

   return (microbenchmarks)
}


#' Initializes the list of default sparse QR factorization microbenchmarks
#'
#' \code{GetSparseQrDefaultMicrobenchmarks} defines the default sparse QR
#' factorization microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The current sparse QR
#' factorization microbenchmarks cover a variety of matrices of different
#' dimensions and number of non-zero values.  They are as follows:
#' \enumerate{
#'   \item Maragal_6 -- rank deficient least squares matrix of 537694 nonzeros
#'   \item landmark -- least squares matrix of 1146848
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmarks appear in the order
#'   listed in the function description and are assigned the names enumerated
#'   in the description.
#' @seealso \code{\link{SparseMatrixMicrobenchmark}}
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseQrDefaultMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["qr_Maragal_6"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "qr_Maragal_6",
      benchmarkDescription = "QR factorization of Maragal_6 matrix from University of Florida Sparse Matrix Collection NYPA group; rank deficient least squares problem, D. Maragal, NY Power Authority",
      matrixObjectName = "Maragal_6",
      numberOfRows = as.integer(21255),
      numberOfColumns = as.integer(10152),
      numberOfNonzeros = as.integer(537694),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseQrAllocator,
      benchmarkFunction = SparseQrMicrobenchmark
   )

   microbenchmarks[["qr_landmark"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "qr_landmark",
      benchmarkDescription = "QR factorization of landmark matrix from University of Florida Sparse Matrix Collection Pereyra group; Matrix from Victor Pereyra, Stanford University",
      matrixObjectName = "landmark",
      numberOfRows = as.integer(71952),
      numberOfColumns = as.integer(2704),
      numberOfNonzeros = as.integer(1146848),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseQrAllocator,
      benchmarkFunction = SparseQrMicrobenchmark
   )

   return (microbenchmarks)
}


#' Initializes the list of example sparse matrix-vector microbenchmarks
#'
#' \code{GetSparseMatrixVectorExampleMicrobenchmarks} defines example sparse
#' matrix-vector microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The example
#' matrix-vector microbenchmarks cover the following matrices:
#' \enumerate{
#'   \item matvec_laplacian7pt_100 -- 100x100x100 7-point Laplacian operator
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmark for matrix-vector
#'   operations with the Laplacian operator laplacian7pt_100 is
#'   returned.
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseMatrixVectorExampleMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["matvec_laplacian7pt_100"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "matvec_laplacian7pt_100",
      benchmarkDescription = "sparse matrix-vector mult. with 100x100x100 7-point Laplacian operator",
      matrixObjectName = "laplacian7pt_100",
      numberOfRows = as.integer(1000000),
      numberOfColumns = as.integer(1000000),
      numberOfNonzeros = as.integer(6940000),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseMatrixVectorAllocator,
      benchmarkFunction = SparseMatrixVectorMicrobenchmark
   )

   return (microbenchmarks)
}


#' Initializes the list of example sparse Cholesky factorization
#' microbenchmarks
#'
#' \code{GetSparseCholeskyExampleMicrobenchmarks} defines the example sparse
#' Cholesky factorization microbenchmarks to be executed by the
#' \code{\link{RunSparseMatrixBenchmark}} function.  The current sparse
#' Cholesky factorization microbenchmarks cover the following matrices:
#' \enumerate{
#'   \item cholesky_ct20stif -- Boeing structural matrix with 2600295 nonzeros
#' }
#' See the documentation for the
#' \code{\link{SparseMatrixMicrobenchmark}} class for more details.
#'
#' @return a list of \code{SparseMatrixMicrobenchmark} objects defining the
#'   microbenchmarks to be executed.  The microbenchmark for the Cholesky
#'   factorization of the ct20stif matrix.
#'
#' @family sparse matrix default microbenchmarks
#' @export
GetSparseCholeskyExampleMicrobenchmarks <- function() {
   microbenchmarks <- list()
   microbenchmarks[["cholesky_ct20stif"]] <- methods::new(
      "SparseMatrixMicrobenchmark",
      active = TRUE,
      benchmarkName = "cholesky_ct20stif",
      benchmarkDescription = "Cholesky factorization of ct20stif matrix from University of Florida Sparse Matrix Collection Boeing group; CT20 engine block structural problem -- stiffness matrix, Boeing",
      matrixObjectName = "ct20stif",
      numberOfRows = as.integer(52329),
      numberOfColumns = as.integer(52329),
      numberOfNonzeros = as.integer(2600295),
      numberOfTrials = as.integer(c(3)),
      numberOfWarmupTrials = as.integer(c(1)),
      allocatorFunction = SparseCholeskyAllocator,
      benchmarkFunction = SparseCholeskyMicrobenchmark
   )

   return (microbenchmarks)
}

