# PeruAPIs - Access Peruvian Data via Public APIs and Curated Datasets
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_peru_hospital_beds


library(testthat)

test_that("get_peru_hospital_beds() returns a tibble with correct structure", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_s3_class(result, "tbl_df")
  expect_named(result, c("indicator", "country", "year", "value"))
  expect_equal(ncol(result), 4)
  expect_equal(nrow(result), 13) # 2010–2022
})

test_that("get_peru_hospital_beds() returns correct column types", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_peru_hospital_beds() returns only Peru and correct indicator", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_true(all(result$country == "Peru"))
  expect_true(all(result$indicator == "Hospital beds (per 1,000 people)"))
})

test_that("get_peru_hospital_beds() covers years 2010 to 2022", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_true(all(result$year %in% 2010:2022))
  expect_equal(length(unique(result$year)), 13)
})

test_that("get_peru_hospital_beds() allows NA values in 'value' column", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  # At least one NA (2022 in console output), but valid if none
  expect_true(any(is.na(result$value)) || all(!is.na(result$value)))
})

test_that("get_peru_hospital_beds() has no NA values in year or country", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_false(any(is.na(result$year)))
  expect_false(any(is.na(result$country)))
})

test_that("get_peru_hospital_beds() returns years sorted in descending order", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_equal(result$year, sort(result$year, decreasing = TRUE))
})

test_that("get_peru_hospital_beds() has consistent indicator and country", {
  skip_on_cran()
  result <- get_peru_hospital_beds()

  expect_equal(length(unique(result$indicator)), 1)
  expect_equal(length(unique(result$country)), 1)
})
