% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluation.R
\name{EvalTest}
\alias{EvalTest}
\alias{WCB_TVTest}
\alias{WCB_HETest}
\title{Evaluate an estimated PSTR model}
\usage{
EvalTest(use, type = c("time-varying", "heterogeneity"), vq = NULL)

WCB_TVTest(use, iB = 100, parallel = FALSE, cpus = 4)

WCB_HETest(use, vq, iB = 100, parallel = FALSE, cpus = 4)
}
\arguments{
\item{use}{An object of class \code{"PSTR"} returned by \code{\link{EstPSTR}}.
The model must be estimated (nonlinear PSTR) before evaluation tests can be run.}

\item{type}{Character vector. Which evaluation tests to run in \code{EvalTest}.
Must be a subset of \code{c("time-varying","heterogeneity")}. Default is both.}

\item{vq}{Numeric vector. Candidate transition variable used by the no remaining nonlinearity
test. Required if \code{"heterogeneity"} is included in \code{type}, and required for
\code{WCB_HETest}.}

\item{iB}{Integer. Number of bootstrap replications. Default is \code{100}.}

\item{parallel}{Logical. Whether to use parallel computation (via the \pkg{snowfall} backend).}

\item{cpus}{Integer. Number of CPU cores used if \code{parallel = TRUE}.}
}
\value{
Invisibly returns \code{use} with evaluation results added.
\describe{
  \item{\code{tv}}{A list of parameter-constancy (time-varying) test results, one element per \eqn{m}.}
  \item{\code{ht}}{A list of no remaining nonlinearity (heterogeneity) test results, one element per \eqn{m}.}
  \item{\code{wcb_tv}}{A numeric matrix of WB/WCB p-values for parameter-constancy tests (one row per \eqn{m}).}
  \item{\code{wcb_ht}}{A numeric matrix of WB/WCB p-values for no remaining nonlinearity tests (one row per \eqn{m}).}
}
The individual list elements in \code{tv} and \code{ht} contain LM-type test statistics and
p-values (including HAC variants), consistent with the output from \code{\link{LinTest}}.
}
\description{
\code{EvalTest} provides post-estimation evaluation tests for an estimated PSTR model.
It supports two null hypotheses:
\describe{
  \item{Parameter constancy}{No time variation in parameters (labelled \code{"time-varying"}).}
  \item{No remaining nonlinearity}{No remaining nonlinearity/heterogeneity given a candidate transition variable (labelled \code{"heterogeneity"}).}
}
}
\details{
Wild bootstrap (WB) and wild cluster bootstrap (WCB) versions are available via
\code{WCB_TVTest} (parameter constancy) and \code{WCB_HETest} (no remaining nonlinearity).

Two equivalent interfaces are available for each test:
\enumerate{
  \item Wrapper function, for example \code{EvalTest(use = obj, ...)}.
  \item R6 method, for example \code{obj$EvalTest(...)}.
}
Each wrapper calls the corresponding R6 method and returns \code{use} invisibly.

The bootstrap variants are computationally intensive. WB is robust to heteroskedasticity,
while WCB is additionally robust to within-individual dependence (cluster dependence).
Parallel execution can be enabled via \code{parallel} and \code{cpus}.
}
\examples{
\donttest{
pstr <- NewPSTR(Hansen99, dep = "inva", indep = 4:20,
               indep_k = c("vala","debta","cfa","sales"),
               tvars = c("vala"), iT = 14)

# estimate first
pstr <- EstPSTR(use = pstr, im = 1, iq = 1, useDelta = TRUE, par = c(.63, 0), method = "CG")

# evaluation tests
pstr <- EvalTest(
  use = pstr,
  type = c("time-varying","heterogeneity"),
  vq = as.matrix(Hansen99[,'vala'])[,1]
)
print(pstr, mode = "evaluation")

# bootstrap variants (requires snowfall)
library(snowfall)
pstr <- WCB_TVTest(
    use = pstr, iB = 4,
    parallel = TRUE, cpus = 2)
pstr <- WCB_HETest(
    use = pstr,
    vq = as.matrix(Hansen99[,'vala'])[,1],
    iB = 4, parallel = TRUE, cpus = 2)
print(pstr, mode = "evaluation")
}

}
\seealso{
\code{\link{NewPSTR}}, \code{\link{LinTest}}, \code{\link{WCB_LinTest}},
  \code{\link{EstPSTR}}.
}
