% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wilcoxon.R
\name{wilcox_test_pv}
\alias{wilcox_test_pv}
\title{Wilcoxon's signed-rank test}
\usage{
wilcox_test_pv(
  x,
  y = NULL,
  mu = 0,
  alternative = "two.sided",
  exact = NULL,
  correct = TRUE,
  digits_rank = Inf,
  simple_output = FALSE
)
}
\arguments{
\item{x}{numerical vector forming the sample to be tested or a list of
numerical vectors for multiple tests.}

\item{y}{numerical vector forming the second sample to be tested or a
list of numerical vectors for multiple tests; if \code{y = NULL}
(the default), the one-sample version is performed; for
two-sample tests, all sample pairs must have the same length.}

\item{mu}{numerical vector of hypothesised location(s) for one-sample
tests or location shift(s) for two-sample tests.}

\item{alternative}{character vector that indicates the alternative hypotheses; each value must be one of \code{"two.sided"} (the default), \code{"less"} or \code{"greater"}.}

\item{exact}{logical value that indicates whether p-values are to be calculated by exact computation (\code{TRUE}; the default) or by a continuous approximation (\code{FALSE}).}

\item{correct}{logical value that indicates if a continuity correction is to be applied (\code{TRUE}; the default) or not (\code{FALSE}). Ignored, if \code{exact = TRUE}.}

\item{digits_rank, }{single number giving the significant digits used to compute ranks for the test statistics.}

\item{simple_output, }{logical value that indicates whether an R6 class object, including the tests' parameters and support sets, i.e. all observable p-values under each null hypothesis, is to be returned (see below).}
}
\value{
If \code{simple.output = TRUE}, a vector of computed p-values is returned.
Otherwise, the output is a \code{\link{DiscreteTestResults}} R6 class object, which
also includes the p-value supports and testing parameters. These have to be
accessed by public methods, e.g. \verb{$get_pvalues()}.
}
\description{
\code{wilcox_test_pv()} performs an exact or approximate Wilcoxon signed-rank test
about the location of a population on a single sample or the differences
between two paired groups when the data is not necessarily normally
distributed. In contrast to \code{\link[stats:wilcox.test]{stats::wilcox.test()}}, it is vectorised and
only calculates \emph{p}-values. Furthermore, it is capable of returning the
discrete \emph{p}-value supports, i.e. all observable \emph{p}-values under a null
hypothesis. Multiple tests can be evaluated simultaneously.
}
\details{
The parameters \code{x}, \code{mu} and \code{alternative} are vectorised. If \code{x} is a
list, they are replicated automatically to have the same lengths. In case \code{x}
is not a list, it is added to one, which is then replicated to the
appropriate length. This allows multiple hypotheses to be tested
simultaneously.

In the presence of ties or observations that are equal to \code{mu},
computation of exact \emph{p}-values is not possible. Therefore, \code{exact} is
ignored in these cases and \emph{p}-values of the respective test settings are
calculated by a normal approximation.

By setting \code{exact = NULL}, exact computation is performed if the sample in a
test setting does not have any ties or zeros and if the sample size is lower
than or equal to 200.

The used test statistics \code{W} is also known as \eqn{T+} and is defined as the
sum of ranks of all strictly positive values of the sample \code{x}.

If \code{digits_rank = Inf} (the default), \code{\link[base:rank]{rank()}} is used to
compute ranks for the tests statistics instead of
\code{\link[base:rank]{rank}}(\code{\link[base:Round]{signif(., digits_rank)}})
}
\examples{
# Constructing
set.seed(1)
r1 <- rnorm(200)
r2 <- rnorm(200, 1)
r3 <- rnorm(200, 2)

## One-sample tests
#  Exact two-sided p-values and their supports
results_ex_1s <- wilcox_test_pv(r1)
print(results_ex_1s)
results_ex_1s$get_pvalues()
results_ex_1s$get_pvalue_supports()

#  Multiple normal-approximated one-sided tests ("greater")
results_ap_1s <- wilcox_test_pv(list(r1, r2), alternative = "greater", exact = FALSE)
print(results_ap_1s)
results_ap_1s$get_pvalues()
results_ap_1s$get_pvalue_supports()

## Two-sample-tests
#  Normal-approximated one-sided p-values ("less") and their supports
results_ap_2s <- wilcox_test_pv(r1, r2, alternative = "less", exact = FALSE)
print(results_ap_2s)
results_ap_2s$get_pvalues()
results_ap_2s$get_pvalue_supports()

#  Multiple exact two-sided tests ("greater")
results_ex_2s <- wilcox_test_pv(list(r1, r3), r2)
print(results_ex_2s)
results_ex_2s$get_pvalues()
results_ex_2s$get_pvalue_supports()

}
\references{
Hollander, M. & Wolfe, D. (1973). \emph{Nonparametric Statistical Methods}. Third
Edition. New York: Wiley. pp. 40-55. \doi{10.1002/9781119196037}
}
\seealso{
\code{\link[stats:wilcox.test]{stats::wilcox.test()}}, \code{\link[=mann_whitney_test_pv]{mann_whitney_test_pv()}}
}
