## ----echo=FALSE---------------------------------------------------------------
knitr::opts_chunk$set(
  comment = ''
)

## -----------------------------------------------------------------------------
library(xdvir)

## ----echo=FALSE---------------------------------------------------------------
if (getOption("xdvir.engine")$name == "null") {
    ## TeX not available
    haveTeX <- FALSE
    warning("TeX not available.  Example code will not be run")
} else {
    haveTeX <- TRUE
    ## Math typesetting by LuaTeX can fall back into mode=base
    ## (which we can't currently handle)
    ## so use XeTeX engine for this vignette
    options("xdvir.engine"="xetex")
}

## ----label="tex", echo=FALSE--------------------------------------------------
tex <- r"(\huge $\Phi(z) = \frac{1}{\sqrt{2\pi}} \cdot e^{-\frac{z^2}{2}}$)"

## ----echo=FALSE, message=FALSE------------------------------------------------
library(gggrid)

## ----gaussian, echo=FALSE-----------------------------------------------------
x <- seq(-4, 4, length.out=100)
df <- data.frame(x=x, y=dnorm(x))

## ----ggbase, echo=FALSE-------------------------------------------------------
gg <- ggplot(df) +
    geom_line(aes(x, y))

## ----echo=FALSE, fig.width=6, fig.height=3, message=FALSE---------------------
if (haveTeX) {
    gg +
    grid_panel(latexGrob(tex, x=0, y=1, hjust="left", vjust="top")) +
    xlim(-4, 4) +
    geom_area(aes(x, y), colour=NA, fill="grey90") +
    geom_line(aes(x, y)) +
    scale_x_continuous(expand=expansion(0)) +
    scale_y_continuous(expand=expansion(c(0))) +
    coord_cartesian(clip="off") +
    theme_minimal() +
    theme(panel.grid=element_blank(),
          axis.title=element_blank(),
          axis.text.y=element_blank(),
          plot.margin=unit(rep(4, 4), "mm")) 
} else {
    grid.text("TeX not available")
}

## -----------------------------------------------------------------------------
tex <- r"(\huge $\Phi(z) = \frac{1}{\sqrt{2\pi}} \cdot e^{-\frac{z^2}{2}}$)"

## ----fig.width=3, fig.height=1, eval=haveTeX----------------------------------
grid.latex(tex)

## ----fig.width=3, fig.height=1, echo=FALSE, eval=!haveTeX---------------------
# grid.text("TeX not available")

## -----------------------------------------------------------------------------
library(lattice)

## ----eval=FALSE---------------------------------------------------------------
# x <- seq(-4, 4, length.out=100)
# df <- data.frame(x=x, y=dnorm(x))

## ----label="lattice-init", echo=FALSE, eval=FALSE-----------------------------
# oopt <- lattice.options(layout.widths=list(left.padding=list(x=0),
#                                            right.padding=list(x=0)),
#                         layout.heights=list(bottom.padding=list(x=0),
#                                             top.padding=list(x=0)))

## ----label="lattice-plot", eval=FALSE-----------------------------------------
# xyplot(y ~ x, df, type="l",
#        panel=function(...) {
#            panel.xyplot(...)
#            grid.latex(tex,
#                       x=unit(2, "mm"),
#                       y=unit(1, "npc") - unit(2, "mm"),
#                       hjust="left", vjust="top")
#        })

## ----label="lattice-final", echo=FALSE, eval=FALSE----------------------------
# lattice.options(oopt)

## ----echo=FALSE, fig.width=6, fig.height=3------------------------------------
if (haveTeX) {
oopt <- lattice.options(layout.widths=list(left.padding=list(x=0),
                                           right.padding=list(x=0)),
                        layout.heights=list(bottom.padding=list(x=0),
                                            top.padding=list(x=0)))
xyplot(y ~ x, df, type="l",
       panel=function(...) {
           panel.xyplot(...)
           grid.latex(tex, 
                      x=unit(2, "mm"),
                      y=unit(1, "npc") - unit(2, "mm"),
                      hjust="left", vjust="top")
       })
lattice.options(oopt)
} else {
    grid.text("TeX not available")
}

## -----------------------------------------------------------------------------
library(gridGraphics)

## ----label="graphics-init", echo=FALSE, eval=FALSE----------------------------
# opar <- par(mar=c(4, 4, 1, 1))

## ----label="graphics-plot", eval=FALSE----------------------------------------
# plot(y ~ x, df, type="l")
# grid.echo()
# downViewport("graphics-plot-1")
# grid.latex(tex,
#            x=unit(2, "mm"),
#            y=unit(1, "npc") - unit(2, "mm"),
#            hjust="left", vjust="top")

## ----label="graphics-final", echo=FALSE, eval=FALSE---------------------------
# par(opar)

## ----echo=FALSE, fig.width=6, fig.height=3, fig.keep="last"-------------------
if (haveTeX) {
opar <- par(mar=c(4, 4, 1, 1))
plot(y ~ x, df, type="l")
grid.echo()
downViewport("graphics-plot-1")
grid.latex(tex, 
           x=unit(2, "mm"),
           y=unit(1, "npc") - unit(2, "mm"),
           hjust="left", vjust="top")
par(opar)
} else {
    grid.text("TeX not available")
}

## ----message=FALSE------------------------------------------------------------
library(gggrid)

## ----eval=FALSE---------------------------------------------------------------
# gg <- ggplot(df) +
#     geom_line(aes(x, y))

## ----fig.width=6, fig.height=3, eval=haveTeX----------------------------------
gg +
    grid_panel(latexGrob(tex,
                         x=unit(2, "mm"), 
                         y=unit(1, "npc") - unit(2, "mm"), 
                         hjust="left", vjust="top"))

## ----fig.width=6, fig.height=3, echo=FALSE, eval=!haveTeX---------------------
# grid.text("TeX not available")

## ----fig.width=6, fig.height=3, message=FALSE, eval=haveTeX-------------------
gg +
    labs(title=paste("The Normal Distribution:", tex)) +
    theme(plot.title=element_latex())

## ----fig.width=6, fig.height=3, message=FALSE, echo=FALSE, eval=!haveTeX------
# grid.text("TeX not available")

## -----------------------------------------------------------------------------
samples <- data.frame(x=rnorm(50), sample=rep(1:5, each=10))
means <- aggregate(samples$x, list(sample=samples$sample), mean)
means$label <- paste0("$\\bar x_", means$sample, "$")

## ----fig.width=6, fig.height=3, message=FALSE, eval=haveTeX-------------------
ggplot(samples) +
    geom_vline(xintercept=0, linetype="solid", colour=1, linewidth=.5) +
    geom_point(aes(x, sample), size=4, alpha=.5) +
    geom_point(aes(x, sample), data=means, colour=2, size=4) +
    geom_latex(aes(x, sample, label=label), data=means, 
               size=6, vjust=-.4, colour=2) +
    scale_y_continuous(expand=expansion(.25))

## ----fig.width=6, fig.height=3, message=FALSE, echo=FALSE, eval=!haveTeX------
# grid.text("TeX not available")

